# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import numpy as np

# ===================
# Part 2: Data Preparation
# ===================
# Data points for art styles, number of exhibitions and popularity score.
art_styles = [
    "Impressionism",
    "Cubism",
    "Surrealism",
    "Abstract",
    "Modern",
    "Renaissance",
]
exhibitions = [50, 30, 20, 15, 10, 5]  # Number of exhibitions held
popularity = [50, 75, 60, 55, 40, 30]  # Popularity score (%)

# Labels and title
xlabel = "Number of Exhibitions"
ylabel = "Popularity Score (%)"
title = "Art Style Popularity vs. Number of Exhibitions"
supertitle = "Art Style Analysis"
legend_labels = art_styles

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and a subplot with a specific size.
fig, ax = plt.subplots(figsize=(8, 5))

# Color scheme reflecting art styles
colors = ["#FFB6C1", "#87CEEB", "#FFD700", "#40E0D0", "#FFA07A", "#9370DB"]
markers = ["o", "v", "^", "<", ">", "*"]

# Scatter plot each art style's data point and add text annotation.
for i in range(len(art_styles)):
    ax.scatter(
        exhibitions[i],
        popularity[i],
        color=colors[i],
        marker=markers[i],
        s=100,
        edgecolor="k",
    )
    ax.text(
        exhibitions[i] + 1,
        popularity[i] + 1,
        art_styles[i],
        fontsize=10,
        ha="left",
        va="bottom",
    )

# Add a horizontal and a vertical dashed reference line.
ax.axhline(
    y=50, color="gray", linestyle="--", linewidth=1
)  # Horizontal line at Popularity=50%
ax.axvline(
    x=15, color="gray", linestyle="--", linewidth=1
)  # Vertical line at Exhibitions=15

# Set the scales of the axes to logarithmic.
ax.set_xscale("log")
ax.set_yscale("log")

# Set the labels for the axes.
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Set the title of the plot.
ax.set_title(title)
fig.suptitle(supertitle, fontsize=14)

# Adding grid
ax.grid(True, which="both", linestyle="--", linewidth=0.5)

# Adding legend
for i, label in enumerate(legend_labels):
    ax.scatter([], [], color=colors[i], marker=markers[i], label=label)
ax.legend(title="Art Styles")

# Define major ticks for both axes.
ax.set_xticks([5, 10, 20, 50, 100])
ax.get_xaxis().set_major_formatter(plt.ScalarFormatter())
ax.set_yticks([30, 40, 50, 60, 75, 90])
ax.get_yaxis().set_major_formatter(plt.ScalarFormatter())

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with a tight layout to ensure all elements fit within the figure area.
plt.tight_layout(rect=[0, 0, 1, 0.95])  # Adjust for supertitle
plt.savefig("scatter_45.pdf", bbox_inches="tight")
