# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Set a random seed for reproducibility

# Generate temperature data
mean_values_A = [0, 10, 25, 15]  # Winter, Spring, Summer, Fall for Location A
mean_values_B = [5, 15, 30, 20]  # Winter, Spring, Summer, Fall for Location B
standard_deviations = [5, 5, 5, 5]

data_A = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_A, standard_deviations)
]
data_B = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_B, standard_deviations)
]
positions_A = np.array(range(1, len(data_A) + 1)) - 0.2
positions_B = np.array(range(1, len(data_B) + 1)) + 0.2
legend_labels = ["Location A", "Location B"]
xlabel = "Season"
ylabel = "Temperature (°C)"
xticks = [1, 2, 3, 4]
xtickslabels = ["Winter", "Spring", "Summer", "Fall"]
title = "Seasonal Temperature Variations"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with the specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Narrower violin plots
violin_width = 0.35  # set this to a suitable value

# Create the violin plot with adjusted positions

parts_A = ax.violinplot(
    data_A,
    positions=positions_A,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_B = ax.violinplot(
    data_B,
    positions=positions_B,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing the colors of the violin plot
colors_A = ["#87CEEB", "#87CEEB", "#87CEEB", "#87CEEB"]  # Light Blue for Winter
colors_B = ["#FF6347", "#FF6347", "#FF6347", "#FF6347"]  # Tomato for Location B

for pc, color in zip(parts_A["bodies"], colors_A):
    pc.set_facecolor(color)
    pc.set_edgecolor("black")
    pc.set_alpha(1)

for pc, color in zip(parts_B["bodies"], colors_B):
    pc.set_facecolor(color)
    pc.set_edgecolor("black")
    pc.set_alpha(1)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_A[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_B[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Customizing the colors of the violin plot and adding statistics
for i in range(len(data_A)):
    # Adding the statistical annotations for data_A
    quartile1, median, quartile3 = np.percentile(data_A[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_A[i][data_A[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_A[i][data_A[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_A[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions_A[i] - 0.025,
        positions_A[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_A[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

    # Adding the statistical annotations for data_B
    quartile1, median, quartile3 = np.percentile(data_B[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_B[i][data_B[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_B[i][data_B[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_B[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions_B[i] - 0.025,
        positions_B[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_B[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_A["bodies"][0], parts_B["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.0),
    ncol=2,
)

# Setting the title and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(
    True, linestyle="-", linewidth=0.7, color="grey", zorder=0
)  # You can customize the style of the grid here

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_22.pdf", bbox_inches="tight")
