
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# New data simulating popularity distribution of different genres
popularities_without_marketing = np.random.beta(a=[2, 1, 1, 3, 2], b=[5, 4, 3, 2, 1], size=(100, 5))
popularities_with_marketing = np.random.beta(a=[4, 3, 3, 5, 4], b=[2, 3, 1, 2, 2], size=(100, 5))

# Y-axis label and limits
ylabel = "Popularity Score"
ylim = [0, 1]
violin_width = 0.5
scaling_factor = 1
kde_x = np.linspace(0, 1, 300)

# Offsets for groups
offsets = np.linspace(-3, 3, 5)
labels = ["Without Marketing", "With Marketing"]
titles = ["Popularity Distribution by Genre"]
legend_labels = ["With Marketing", "Without Marketing"]
xticklabels = ["Action", "Comedy", "Drama", "Horror", "Sci-fi"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure
fig, ax = plt.subplots(1, 1, figsize=(10, 6))

# Define the colors for the two distributions
colors = ["#1f77b4", "#ff7f0e"]

# Function to plot half violins
def plot_half_violin(ax, data, data_memory, offsets, colors, labels, title, xticklabels):
    for i in range(data.shape[1]):
        offset = offsets[i]

        # Plot data without marketing
        kde_data = gaussian_kde(data[:, i])
        kde_x = np.linspace(0, 1, 300)
        kde_data_y = kde_data(kde_x)
        kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            kde_data_y_scaled * scaling_factor + offset,
            offset,
            color=colors[0],
            edgecolor='#2a9df4',
            alpha=0.7
        )
        
        # Plot data with marketing
        kde_data_memory = gaussian_kde(data_memory[:, i])
        kde_data_memory_y = kde_data_memory(kde_x)
        kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            offset,
            -kde_data_memory_y_scaled * scaling_factor + offset,
            color=colors[1],
            edgecolor='#ff9731',
            alpha=0.7
        )

    # Set x and y axis labels, limits, and add x-axis tick labels for groups
    ax.set_xlim(
        min(offsets) - scaling_factor - violin_width,
        max(offsets) + scaling_factor + violin_width,
    )
    ax.set_ylim(ylim)
    ax.set_ylabel(ylabel)
    ax.set_xticks(offsets)
    ax.set_xticklabels(xticklabels)
    ax.title.set_text(title)

# Plot the violins
plot_half_violin(ax, popularities_without_marketing, popularities_with_marketing, offsets, colors, labels, titles[0], xticklabels)

# Add a legend for the entire figure
handles = [
    plt.Line2D([0], [0], marker='o', color='w', markerfacecolor=colors[1], markersize=10),
    plt.Line2D([0], [0], marker='o', color='w', markerfacecolor=colors[0], markersize=10),
]

fig.legend(handles, legend_labels, loc='lower center', bbox_to_anchor=(0.5, -0.1), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Tighten the layout and show the combined plot
plt.tight_layout()
plt.savefig("violin_27.pdf", bbox_inches="tight")
