# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Sample financial data for demonstration purposes
market_conditions = np.arange(1, 6)
strategy_a_data = (
    np.random.randn(10, 100) * 10 + 100
)  # Centered around 100 with some noise
strategy_b_data = (
    np.random.randn(10, 100) * 12 + 110
)  # Centered around 110 with some noise

titles = [
    "% Change in Portfolio Value vs. Strategy A",
    "% Change in Portfolio Value vs. Strategy B",
]
ylims = [[50, 150], [50, 150]]
xlabel = "Market Conditions"
ytickslabels = [
    ["50%", "75%", "100%", "125%", "150%"],
    ["50%", "75%", "100%", "125%", "150%"],
]
xticklabel = ["Condition 1", "Condition 2", "Condition 3", "Condition 4", "Condition 5"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and subplots
fig, axs = plt.subplots(2, 1, figsize=(10, 10))

# Define the colors for the violin plots
color_strategy_a = "#228B22"  # ForestGreen for Strategy A
color_strategy_b = "#4682B4"  # SteelBlue for Strategy B


# Function to set the color of the violin plot
def set_violin_color(violin, color):
    for body in violin["bodies"]:
        body.set_facecolor(color)
        body.set_edgecolor(color)
    # Set color for the median line
    violin["cmedians"].set_color(color)
    violin["cmedians"].set_linewidth(2)


# Top subplot: Strategy A
for i, condition in enumerate(market_conditions):
    vl = axs[0].violinplot(
        strategy_a_data[i],
        positions=[condition - 0.2],
        showmedians=True,
        widths=0.3,
        showextrema=False,
    )
    set_violin_color(vl, color_strategy_a)

    vl = axs[0].violinplot(
        strategy_b_data[i],
        positions=[condition + 0.2],
        showmedians=True,
        widths=0.3,
        showextrema=False,
    )
    set_violin_color(vl, color_strategy_b)

    axs[0].text(
        condition - 0.35,
        np.median(strategy_a_data[i]),
        f"{int(np.median(strategy_a_data[i]))}%",
        ha="right",
        va="bottom",
        color=color_strategy_a,
    )
    axs[0].text(
        condition + 0.35,
        np.median(strategy_b_data[i]),
        f"{int(np.median(strategy_b_data[i]))}%",
        ha="left",
        va="bottom",
        color=color_strategy_b,
    )

axs[0].set_title(titles[0])
axs[0].set_xticks(market_conditions)
axs[0].set_ylim(ylims[0])
axs[0].set_xticklabels([])
axs[0].set_yticklabels(ytickslabels[0])

# Bottom subplot: Strategy B
for i, condition in enumerate(market_conditions):
    vl = axs[1].violinplot(
        strategy_a_data[i],
        positions=[condition - 0.2],
        showmedians=True,
        widths=0.3,
        showextrema=False,
    )
    set_violin_color(vl, color_strategy_a)

    vl = axs[1].violinplot(
        strategy_b_data[i],
        positions=[condition + 0.2],
        showmedians=True,
        widths=0.3,
        showextrema=False,
    )
    set_violin_color(vl, color_strategy_b)

    axs[1].text(
        condition - 0.35,
        np.median(strategy_a_data[i]),
        f"{int(np.median(strategy_a_data[i]))}%",
        ha="right",
        va="bottom",
        color=color_strategy_a,
    )
    axs[1].text(
        condition + 0.35,
        np.median(strategy_b_data[i]),
        f"{int(np.median(strategy_b_data[i]))}%",
        ha="left",
        va="bottom",
        color=color_strategy_b,
    )

axs[1].set_title(titles[1])
axs[1].set_xticks(market_conditions)
axs[1].set_ylim(ylims[1])
axs[1].set_xlabel(xlabel)
axs[1].set_xticklabels(xticklabel)
axs[1].set_yticklabels(ytickslabels[1])

# use line to create a custom legend
legend_elements = [
    Line2D([0], [0], color=color_strategy_a, lw=2, label="Strategy A"),
    Line2D([0], [0], color=color_strategy_b, lw=2, label="Strategy B"),
]
axs[0].legend(handles=legend_elements, loc="upper right")
axs[1].legend(handles=legend_elements, loc="upper right")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout()
plt.savefig("violin_40.pdf", bbox_inches="tight")
