
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Generate synthetic temperature data for different climates
np.random.seed(0)

# Define mean and std dev for different climates
tropical_data = np.clip(np.random.normal(30, 4, 200), 15, 45)
dry_data = np.clip(np.random.normal(25, 5, 200), 10, 40)
temperate_data = np.clip(np.random.normal(20, 6, 200), 5, 35)
continental_data = np.clip(np.random.normal(15, 8, 200), -5, 30)
polar_data = np.clip(np.random.normal(0, 10, 200), -30, 10)

data = [tropical_data, dry_data, temperate_data, continental_data, polar_data]
categories = ["Tropical", "Dry", "Temperate", "Continental", "Polar"]
ylabel = "Temperature (°C)"
ylim = [-35, 50]
xlabel = "Climate Types"
textlabels = ["Mean Temp", "Temp Range"]

# Calculate mean and temperature range for annotations
mean_temp = [np.mean(d) for d in data]
temp_range = [np.ptp(d) for d in data]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Create violin plots
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customize the appearance
ax.set_ylabel(ylabel)
ax.set_xticks(np.arange(1, len(categories) + 1))
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.set_xlabel(xlabel)

# Customize colors and annotations
colors = ["#FF5733", "#FF8D1A", "#FFC300", "#DAF7A6", "#AED6F1"]
for i, (pc, d) in enumerate(zip(violin_parts["bodies"], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor("black")
    pc.set_alpha(1)

    # Calculate quartiles and median
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate whiskers
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Annotate statistics
    ax.vlines(i + 1, quartile1, quartile3, color="k", linestyle="-", lw=4)
    ax.scatter(i + 1, median, color="w", s=10, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color="k", linestyle="-", lw=1)
    ax.text(i + 1 + 0.3, median, f"{median:.2f}", ha="left", va="center", color="black", rotation=45)

    # Annotate with Mean Temp and Temp Range
    ax.text(i + 1, -20, f"{mean_temp[i]:.1f}", ha="center", va="center", color="green", fontsize=10)
    ax.text(i + 1, -30, f"{temp_range[i]:.1f}", ha="center", va="center", color="blue", fontsize=10)

ax.text(5.6, -20, textlabels[0], ha="left", va="center", color="green", fontsize=10)
ax.text(5.6, -30, textlabels[1], ha="left", va="center", color="blue", fontsize=10)

# Make other parts of the violin plots invisible
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_42.pdf", bbox_inches="tight")
