
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import numpy as np

# ===================
# Part 2: Data Preparation
# ===================
# Setting a random seed for reproducibility
np.random.seed(42)

# Generate mean test scores for 5 semesters for 2 classes
mean_scores_class1 = np.linspace(70, 85, 5)
mean_scores_class2 = np.linspace(60, 80, 5)
# Smaller standard deviations for realistic score clustering
standard_deviations = [10] * 5

# Generate data for Class1 and Class2
class1_scores = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_scores_class1, standard_deviations)
]
class2_scores = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_scores_class2, standard_deviations)
]
positions_class1 = np.array(range(1, len(class1_scores) + 1)) - 0.2
positions_class2 = np.array(range(1, len(class2_scores) + 1)) + 0.2

legend_labels = ["Class 1", "Class 2"]
xlabel = "Semester"
ylabel = "Test Scores"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Sem 1", "Sem 2", "Sem 3", "Sem 4", "Sem 5"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Violin plot width
violin_width = 0.35

# Create the violin plot with adjusted positions
parts_class1 = ax.violinplot(
    class1_scores,
    positions=positions_class1,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_class2 = ax.violinplot(
    class2_scores,
    positions=positions_class2,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing the colors of the violin plot
for pc in parts_class1["bodies"]:
    pc.set_facecolor("royalblue")  # Class 1 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

for pc in parts_class2["bodies"]:
    pc.set_facecolor("mediumseagreen")  # Class 2 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_class1[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_class2[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Adding statistics annotations for both classes
for i in range(len(class1_scores)):
    # Class 1 statistics
    quartile1, median, quartile3 = np.percentile(class1_scores[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(class1_scores[i][class1_scores[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(class1_scores[i][class1_scores[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_class1[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_class1[i] - 0.025,
        positions_class1[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_class1[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    # Class 2 statistics
    quartile1, median, quartile3 = np.percentile(class2_scores[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(class2_scores[i][class2_scores[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(class2_scores[i][class2_scores[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_class2[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_class2[i] - 0.025,
        positions_class2[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_class2[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_class1["bodies"][0], parts_class2["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting the title and labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_54.pdf", bbox_inches="tight")
