
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Setting a random seed for reproducibility
import numpy as np

np.random.seed(0)

# Generate mean rainfall (in mm) for 5 months for 2 regions
mean_rainfall_region1 = np.linspace(50, 150, 5)
mean_rainfall_region2 = np.linspace(60, 120, 5)
standard_deviations = [20] * 5

# Generate data for Region1 and Region2
region1_rainfall = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_rainfall_region1, standard_deviations)
]
region2_rainfall = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_rainfall_region2, standard_deviations)
]
positions_region1 = np.array(range(1, len(region1_rainfall) + 1)) - 0.2
positions_region2 = np.array(range(1, len(region2_rainfall) + 1)) + 0.2
legend_labels = ["Region1", "Region2"]
xlabel = "Month"
ylabel = "Average Rainfall (mm)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Jan", "Feb", "Mar", "Apr", "May"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Violin plot width
violin_width = 0.35

# Create the violin plot with adjusted positions
parts_region1 = ax.violinplot(
    region1_rainfall,
    positions=positions_region1,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_region2 = ax.violinplot(
    region2_rainfall,
    positions=positions_region2,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing the colors of the violin plot
for pc in parts_region1["bodies"]:
    pc.set_facecolor("deepskyblue")  # Region1 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

for pc in parts_region2["bodies"]:
    pc.set_facecolor("lightseagreen")  # Region2 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_region1[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_region2[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Adding statistics annotations for both regions
for i in range(len(region1_rainfall)):
    # Region1 statistics
    quartile1, median, quartile3 = np.percentile(region1_rainfall[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(region1_rainfall[i][region1_rainfall[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(region1_rainfall[i][region1_rainfall[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_region1[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_region1[i] - 0.025,
        positions_region1[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_region1[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    # Region2 statistics
    quartile1, median, quartile3 = np.percentile(region2_rainfall[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(region2_rainfall[i][region2_rainfall[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(region2_rainfall[i][region2_rainfall[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_region2[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_region2[i] - 0.025,
        positions_region2[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_region2[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_region1["bodies"][0], parts_region2["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting the title and labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_55.pdf", bbox_inches="tight")
