
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Setting a random seed for reproducibility
import numpy as np

np.random.seed(0)
# Generate mean quarterly revenues for 2 companies over a year
mean_revenues_company1 = np.linspace(650, 750, 5)
mean_revenues_company2 = np.linspace(520, 670, 5)
# Smaller standard deviations for realistic revenue clustering
standard_deviations = [50] * 5

# Generate data for Company1 and Company2
company1_revenues = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_revenues_company1, standard_deviations)
]
company2_revenues = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_revenues_company2, standard_deviations)
]
positions_company1 = np.array(range(1, len(company1_revenues) + 1)) - 0.2
positions_company2 = np.array(range(1, len(company2_revenues) + 1)) + 0.2
legend_labels = ["Company1", "Company2"]
xlabel = "Quarter"
ylabel = "Revenue ($ in millions)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Q1", "Q2", "Q3", "Q4", "Q5"]
title = "Quarterly Revenue Comparison"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Violin plot width
violin_width = 0.35

# Create the violin plot with adjusted positions
parts_company1 = ax.violinplot(
    company1_revenues,
    positions=positions_company1,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_company2 = ax.violinplot(
    company2_revenues,
    positions=positions_company2,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing the colors of the violin plot
for pc in parts_company1["bodies"]:
    pc.set_facecolor("forestgreen")  # Company1 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

for pc in parts_company2["bodies"]:
    pc.set_facecolor("royalblue")  # Company2 color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_company1[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_company2[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Adding statistics annotations for both companies
for i in range(len(company1_revenues)):
    # Company1 statistics
    quartile1, median, quartile3 = np.percentile(company1_revenues[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(company1_revenues[i][company1_revenues[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(company1_revenues[i][company1_revenues[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_company1[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_company1[i] - 0.025,
        positions_company1[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_company1[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    # Company2 statistics
    quartile1, median, quartile3 = np.percentile(company2_revenues[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(company2_revenues[i][company2_revenues[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(company2_revenues[i][company2_revenues[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_company2[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_company2[i] - 0.025,
        positions_company2[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_company2[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_company1["bodies"][0], parts_company2["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.0),
    ncol=2,
)

# Setting the title and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_56.pdf", bbox_inches="tight")
