
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate mean values for battery life (in hours) for Streaming and Gaming
mean_values_streaming = np.linspace(6, 8, 5)
mean_values_gaming = np.linspace(4, 6, 5)

# Standard deviations
standard_deviations_streaming = [1] * 5
standard_deviations_gaming = [2] * 5

# Generate synthetic data for Streaming
data_streaming = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_streaming, standard_deviations_streaming)
]

# Generate synthetic data for Gaming
data_gaming = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_gaming, standard_deviations_gaming)
]

# Set positions for violins
positions_streaming = np.array(range(1, len(data_streaming) + 1)) - 0.2
positions_gaming = np.array(range(1, len(data_gaming) + 1)) + 0.2

# Labels and ticks
legend_labels = ["Streaming", "Gaming"]
xlabel = "Usage Patterns"
ylabel = "Battery Life (Hours)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Sun", "Mon", "Tue", "Wed", "Thu"]
# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(8, 7))

violin_width = 0.35

# Create violin plots
parts_streaming = ax.violinplot(
    data_streaming,
    positions=positions_streaming,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_gaming = ax.violinplot(
    data_gaming,
    positions=positions_gaming,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing colors
for pc in parts_streaming["bodies"]:
    pc.set_facecolor("lightgreen")
    pc.set_edgecolor("black")
    pc.set_alpha(0.8)

for pc in parts_gaming["bodies"]:
    pc.set_facecolor("lightcoral")
    pc.set_edgecolor("black")
    pc.set_alpha(0.8)

# Customizing median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp_streaming = parts_streaming[partname]
    vp_gaming = parts_gaming[partname]
    vp_streaming.set_edgecolor("black")
    vp_gaming.set_edgecolor("black")
    vp_streaming.set_linewidth(1)
    vp_gaming.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp_streaming.set_visible(False)  # Hide caps
        vp_gaming.set_visible(False)  # Hide caps

# Adding statistical annotations
for i in range(len(data_streaming)):
    quartile1, median, quartile3 = np.percentile(data_streaming[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_streaming[i][data_streaming[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_streaming[i][data_streaming[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_streaming[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_streaming[i] - 0.025,
        positions_streaming[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_streaming[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    quartile1, median, quartile3 = np.percentile(data_gaming[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_gaming[i][data_gaming[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_gaming[i][data_gaming[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_gaming[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_gaming[i] - 0.025,
        positions_gaming[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_gaming[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Customize borders
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove ticks
ax.tick_params(axis="both", which="both", length=0)

# Adding legend
ax.legend(
    [parts_streaming["bodies"][0], parts_gaming["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(7, 5)

plt.tight_layout()
plt.savefig("violin_59.pdf", bbox_inches="tight")
