# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Simulated stress level data for two groups across five stages
data_5_stages_group_A = np.random.normal(
    loc=[5, 4, 3, 2, 1], scale=[1, 1.2, 1, 0.9, 0.8], size=(100, 5)
)
data_5_stages_group_B = np.random.normal(
    loc=[6, 5, 4, 3, 2], scale=[1.1, 1, 0.9, 0.8, 0.7], size=(100, 5)
)

ylabel = "Stress Level"
ylim = [0, 8]
violin_width = 0.4
scaling_factor = 1
kde_x = np.linspace(0, 8, 300)

# Offsets for stages
offsets_5_stages = np.linspace(-2, 2, 5)
labels = ["Intervention Group", "Control Group"]
title = "Stress Level Distribution Across Stages of Psychological Intervention"
legend_labels = ["Intervention Group", "Control Group"]
xticklabels = ["Stage 1", "Stage 2", "Stage 3", "Stage 4", "Stage 5"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure
fig, ax = plt.subplots(figsize=(10, 6))

# Define the colors for each stage
colors = ["#88c999", "#8dbdc7", "#bc84cc", "#de94a1", "#e4b162"]


# Function to plot half violins for stress level data
def plot_half_violins(
    ax, data_group_A, data_group_B, offsets, colors, labels, title, xticklabels
):
    # Plot the half-violins with an offset for stages
    for i in range(data_group_A.shape[1]):
        offset = offsets[i]

        # Plot data for Group A
        kde_data_A = gaussian_kde(data_group_A[:, i])
        kde_x = np.linspace(0, 8, 300)
        kde_data_A_y = kde_data_A(kde_x)
        kde_data_A_y_scaled = kde_data_A_y / max(kde_data_A_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            kde_data_A_y_scaled * scaling_factor + offset,
            offset,
            color=colors[i],
            edgecolor="black",
            alpha=0.6,
        )

        # Plot data for Group B
        kde_data_B = gaussian_kde(data_group_B[:, i])
        kde_data_B_y = kde_data_B(kde_x)
        kde_data_B_y_scaled = kde_data_B_y / max(kde_data_B_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            offset,
            -kde_data_B_y_scaled * scaling_factor + offset,
            color=colors[i],
            edgecolor="black",
            alpha=0.3,
        )

    # Set x and y axis labels, limits, and add x-axis tick labels for stages
    ax.set_xlim(
        min(offsets) - scaling_factor - violin_width,
        max(offsets) + scaling_factor + violin_width,
    )
    ax.set_ylim(ylim)
    ax.set_ylabel(ylabel)
    ax.set_xticks(offsets)
    ax.set_xticklabels(xticklabels)
    ax.title.set_text(title)


# Plot the violins
plot_half_violins(
    ax,
    data_5_stages_group_A,
    data_5_stages_group_B,
    offsets_5_stages,
    colors,
    labels,
    title,
    xticklabels,
)

# Add a legend for the entire figure
handles = [
    plt.Line2D(
        [0],
        [0],
        marker="o",
        color="w",
        markerfacecolor="black",
        markersize=10,
        alpha=0.6,
    ),
    plt.Line2D(
        [0],
        [0],
        marker="o",
        color="w",
        markerfacecolor="black",
        markersize=10,
        alpha=0.3,
    ),
]

fig.legend(
    handles, legend_labels, loc="lower center", bbox_to_anchor=(0.5, -0.1), ncol=2
)

# ===================
# Part 4: Saving Output
# ===================
# Tighten the layout and show the combined plot
plt.tight_layout()

# Display the plot
plt.savefig("violin_67.pdf", bbox_inches="tight")
