# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
import random

# New domain: Vitamin Intake Across Different Food Groups
data = np.random.beta(a=[2, 4, 6, 3, 5], b=[5, 3, 4, 7, 6], size=(10, 5))
data_historical = np.random.beta(a=[6, 5, 3, 4, 2], b=[4, 6, 7, 5, 3], size=(40, 5))

xticklabels = ["Fruits", "Vegetables", "Dairy", "Meat", "Grains"]
legend_labels = ["Current Data", "Historical Data"]
scaling_factor = 1
violin_width = 0.5
xlabel = "Food Group"
ylabel = "Vitamin Intake (mg)"

# Adjust the offsets for 5 groups instead of 3
offsets = np.linspace(-3, 3, 5)


supertitle = "Vitamin Intake Across Different Food Groups"
title = "Comparison of Current and Historical Data"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(10, 6))  # Increased width for better spacing

# Define the colors for each group
colors = ["#2ca02c", "#1f77b4"]  # Green and Blue for health and nutrition
legend_colors = ["#2ca02c", "#1f77b4"]

# Plot the half-violins with an offset for 5 groups
for i in range(data.shape[1]):
    offset = offsets[i]

    # Plot current data
    kde_data = gaussian_kde(data[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
    )

    # Plot historical data
    kde_data_historical = gaussian_kde(data_historical[:, i])
    kde_data_historical_y = kde_data_historical(kde_x)
    kde_data_historical_y_scaled = (
        kde_data_historical_y / max(kde_data_historical_y) * violin_width
    )
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_historical_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
    )

    # Add markers for emphasis
    ax.scatter(
        offset,
        random.uniform(0.2, 0.8),
        marker="o",
        color="red",
        s=100,
        zorder=3,
        edgecolor="black",
    )

# Set axis labels and limits
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_xlim(
    min(offsets) - scaling_factor - violin_width,
    max(offsets) + scaling_factor + violin_width,
)
ax.set_xticks(offsets)
ax.set_xticklabels(xticklabels)

# Chart title and subtitle
fig.suptitle(supertitle, fontsize=15)
ax.set_title(title, fontsize=12)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black")
    for color in legend_colors
]
ax.legend(handles, legend_labels, loc="upper right", ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout(rect=[0, 0, 1, 0.95])  # Adjust layout to leave space for suptitle
plt.savefig("violin_72.pdf", bbox_inches="tight")
