
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Simulate new data representing entertainment platform metrics
ent_data1 = np.clip(np.random.normal(0.78, 0.08, 200), 0, 1)  # Streaming Service
ent_data2 = np.clip(np.random.normal(0.82, 0.10, 200), 0, 1)  # Gaming Console
ent_data3 = np.clip(np.random.normal(0.75, 0.09, 200), 0, 1)  # Music Platform
ent_data4 = np.clip(np.random.normal(0.70, 0.11, 200), 0, 1)  # Social Media
ent_data5 = np.clip(np.random.normal(0.65, 0.12, 200), 0, 1)  # VR Experience

# Simulated metrics for User Satisfaction and Engagement Rate
user_satisfaction = [0.85, 0.88, 0.80, 0.74, 0.68]
engagement_rate = [65, 75, 50, 45, 40]

data = [ent_data1, ent_data2, ent_data3, ent_data4, ent_data5]
categories = ["Streaming Service", "Gaming Console", "Music Platform", "Social Media", "VR Experience"]
ylabel = "Entertainment Platform Metrics"
ylim = [0, 1.06]
xlabel = "Platform Type"
textlabels = ["User Satisfaction", "Engagement Rate (%)"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Create violin plots
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customize the appearance
ax.set_ylabel(ylabel)
ax.set_xticks(np.arange(1, len(categories) + 1))
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.set_xlabel(xlabel)

# Define a vibrant entertainment-oriented color palette
colors = ["#e41a1c", "#377eb8", "#4daf4a", "#984ea3", "#ff7f00"]

for i, (pc, d) in enumerate(zip(violin_parts["bodies"], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor("black")
    pc.set_alpha(0.75)
    
    # Calculate the quartiles and median
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate whiskers
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Annotate statistics
    ax.vlines(i + 1, quartile1, quartile3, color="k", linestyle="-", lw=4)
    ax.scatter(i + 1, median, color="w", s=40, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color="k", linestyle="-", lw=1)
    ax.text(i + 1 + 0.3, np.median(data[i]), f"{median:.2f}", ha="left", va="center", color="black", rotation=45)

    # Annotate with User Satisfaction and Engagement Rate values
    ax.text(i + 1, 0.14, f"{user_satisfaction[i]:.2f}", ha="center", va="center", color="green", fontsize=10)
    ax.text(i + 1, 0.08, f"{engagement_rate[i]:.2f}", ha="center", va="center", color="blue", fontsize=10)

ax.text(5.6, 0.14, textlabels[0], ha="left", va="center", color="green", fontsize=10)
ax.text(5.6, 0.08, textlabels[1], ha="left", va="center", color="blue", fontsize=10)

# Make the other parts of the violin plots invisible
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# Add grid for better readability
ax.grid(True, linestyle="--", which="both", color="grey", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("violin_79.pdf", bbox_inches="tight")
