

# ===================
# Part 1: Importing Libraries
# ===================
import math
import matplotlib.pyplot as plt
from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
# Set a random seed for reproducibility
import numpy as np

np.random.seed(0)

def polygon_under_graph(x, y):
    """
    Construct the vertex list which defines the polygon filling the space under
    the (x, y) line graph. This assumes x is in ascending order.
    """
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]

# Define the x-axis data
x = np.linspace(0.0, 10.0, 100)  # Increased resolution
source_ids = range(1, 4)

# New function to simulate pollution levels
def pollution_levels(x, source_id):
    period = 10 / source_id  # Different periodicity for each source
    amplitude = np.random.uniform(0.5, 1.5)  # Random amplitude for variability
    return amplitude * np.sin(2 * np.pi * x / period) + amplitude

# Generate vertices for polygons
verts = [polygon_under_graph(x, pollution_levels(x, source_id)) for source_id in source_ids]

# Extracted variables
x_label = "Time (Years)"
y_label = "Pollution Source ID"
z_label = "Pollution Levels"
title = "Pollution Levels Over Time"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and 3D axis
yticks = [1, 2, 3]
xlim = (0, 10)
ylim = (1, 4)
zlim = (0, 3)
aspect_ratio = [2, 1, 1]
view_elev = 25
view_azim = 135

fig = plt.figure(figsize=(12, 8))
ax = fig.add_subplot(projection="3d")

# Define the face colors
facecolors = plt.get_cmap("coolwarm")(np.linspace(0, 1, len(verts)))

# Create a PolyCollection object
poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax.add_collection3d(poly, zs=source_ids, zdir="y")

# Set the axis labels and limits
ax.set(xlim=xlim, ylim=ylim, zlim=zlim, xlabel=x_label, ylabel=y_label, zlabel=z_label)

# Add ticks and title
ax.set_yticks(yticks)
ax.set_title(title, pad=20)

# Adjust the aspect ratio and view angle
ax.set_box_aspect(aspect_ratio)
ax.view_init(elev=view_elev, azim=view_azim)

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the figure
plt.tight_layout()
plt.savefig("3d_78.pdf", bbox_inches="tight")

