
# ===================
# Part 1: Importing Libraries
# ===================
import math
import matplotlib.pyplot as plt
from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
# Import numpy
import numpy as np

# Set a random seed for reproducibility
np.random.seed(0)

def polygon_under_graph(x, y):
    """
    Construct the vertex list which defines the polygon filling the space under
    the (x, y) line graph. This assumes x is in ascending order.
    """
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]

# Define the x-axis data
x = np.linspace(1, 12, 12)  # 12 months
years = range(2019, 2022)

# New function to simulate number of tourists over varying months and years
def tourists_per_month(x, year):
    # simulate a sinusoidal response to months, modulated by year
    return (2000 + 500 * np.sin(2 * np.pi * (x - 1) / 12)) * (1 + 0.1 * (year - 2019))

# Generate vertices for polygons
verts = [polygon_under_graph(x, tourists_per_month(x, year)) for year in years]

# Extracted variables
x_label = "Month"
y_label = "Year"
z_label = "Number of Tourists"
title = "Tourist Flow by Month and Year"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and 3D axis
yticks = [2019, 2020, 2021]
xlim = (1, 12)
ylim = (2019, 2022)
zlim = (0, 3500)
aspect_ratio = [2, 1, 1]
view_elev = 30
view_azim = 150

fig = plt.figure(figsize=(10, 8))
ax = fig.add_subplot(projection="3d")

# Define the face colors
facecolors = ["#8FBC8F", "#556B2F", "#8B4513"]  # Colors reflecting the original theme

# Create a PolyCollection object
poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax.add_collection3d(poly, zs=years, zdir="y")

# Set the axis labels and limits
ax.set(xlim=xlim, ylim=ylim, zlim=zlim, xlabel=x_label, ylabel=y_label, zlabel=z_label)

# Add ticks and title
ax.set_yticks(yticks)
ax.set_title(title, pad=20)

# Adjust the aspect ratio and view angle
ax.set_box_aspect(aspect_ratio)
ax.view_init(elev=view_elev, azim=view_azim)

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the figure
plt.tight_layout()
plt.savefig("3d_79.pdf", bbox_inches="tight")
