

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate new data simulating systolic blood pressure for Young Adults and Senior Adults
data_young_adults = np.random.normal(loc=120, scale=15, size=100)
data_senior_adults = np.random.normal(loc=140, scale=20, size=100)

# Combine data into a list
data = [data_young_adults, data_senior_adults]

# Create positions for each box plot
positions = [0, 1]
xticks = ["Young Adults", "Senior Adults"]
xlabel = "Age Groups"
ylabel = "Systolic Blood Pressure (mmHg)"
title = "Systolic Blood Pressure Distribution Among Age Groups"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
plt.figure(figsize=(10, 6))

# Define colors for the plots
colors = ["steelblue", "darkturquoise"]

# Calculate the kernel density estimate and plot the density plot for each dataset
for i, d in enumerate(data):
    # Calculate KDE
    kde = gaussian_kde(d)
    # Create an array of values covering the entire range of data for KDE
    kde_x = np.linspace(min(d), max(d), 300)
    kde_y = kde(kde_x)
    # Scale KDE values to match the position of the boxplot
    kde_y_scaled = kde_y / kde_y.max() * 0.1  # Scale factor can be adjusted
    # Plot filled density plot to the left of the boxplot
    offset = 0.2
    plt.fill_betweenx(
        kde_x,
        positions[i] - kde_y_scaled - offset,
        positions[i] - offset,
        alpha=0.5,
        color=colors[i],
        edgecolor="black",
    )

# Create box plots inside the violin plots
for i, d in enumerate(data):
    plt.boxplot(
        d,
        positions=[positions[i]],
        widths=0.05,
        patch_artist=True,
        medianprops=dict(color="black"),
        boxprops=dict(facecolor="none", color="black"),
    )

# Add scatter plot for individual data points
for i, d in enumerate(data):
    x = np.random.normal(positions[i], 0.04, size=len(d))
    plt.scatter(
        x,
        d,
        alpha=0.6,
        color=colors[i],
        edgecolors="w",
        s=20,
    )

# Set the x-axis labels and add title
plt.xticks([0, 1], xticks)
plt.xlabel(xlabel)
plt.title(title)

# Set the y-axis label
plt.ylabel(ylabel)

# Set a linear scale for the y-axis
plt.yscale('linear')

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_107.pdf", bbox_inches="tight")

