

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate new data simulating response times of communication protocols
data_protocol1 = np.random.normal(100, 10, 100) # Protocol 1 response times
data_protocol2 = np.random.normal(110, 15, 100) # Protocol 2 response times

# Combine data into a list
data = [data_protocol1, data_protocol2]

# Create positions for each box plot
positions = [0, 1]
xticks = ["Protocol 1", "Protocol 2"]
xlabel = "Communication Protocols"
ylabel = "Response Time (ms)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(8, 6))

# Set new colors
colors = ["mediumblue", "forestgreen"]
for i, d in enumerate(data):
    kde = gaussian_kde(d)
    kde_x = np.linspace(min(d), max(d), 300)
    kde_y = kde(kde_x)
    kde_y_scaled = kde_y / kde_y.max() * 0.1  # Scale factor can be adjusted
    offset = 0.2
    plt.fill_betweenx(
        kde_x,
        positions[i] - kde_y_scaled - offset,
        positions[i] - offset,
        alpha=0.5,
        color=colors[i],
        edgecolor="black",
    )

# Create box plots inside the violin plots
for i, d in enumerate(data):
    plt.boxplot(
        d,
        positions=[positions[i]],
        widths=0.05,
        patch_artist=True,
        medianprops=dict(color="black"),
        boxprops=dict(facecolor="none", color="black"),
    )

# Add scatter plot for individual data points with grey color
for i, d in enumerate(data):
    x = np.random.normal(positions[i], 0.04, size=len(d))
    plt.scatter(
        x,
        d,
        alpha=0.3,
        color=colors[i],
    )

# Set the x-axis labels and add title
plt.xticks(positions, xticks)
plt.xlabel(xlabel)

# Set the y-axis label
plt.ylabel(ylabel)

# Adjust the y-axis limits for better data distribution visualization
plt.ylim(60, 160)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_108.pdf", bbox_inches="tight")

