
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
from scipy.stats import gaussian_kde

# Generating sample data representing GDP growth rates in three different periods
gdp_early = np.random.normal(2.5, 0.5, 100)  # Early 2000s trend
gdp_mid = np.random.normal(3.0, 0.7, 100)  # Mid 2010s trend
gdp_recent = np.random.normal(1.8, 0.6, 100)  # Recent trend

# Creating KDE for each sample data set to estimate the density of data points
kde_early = gaussian_kde(gdp_early)
kde_mid = gaussian_kde(gdp_mid)
kde_recent = gaussian_kde(gdp_recent)

gdp_range = np.linspace(0, 5, 50)  # Defining the range of GDP growth rates for plotting
labels = ["Early 2000s", "Mid 2010s", "Recent Years"]
ax1_legend_title = "Period"
ax1_xlabel = "Density"
ax2_ylabel = "GDP Growth Rate (%)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Setting up the figure and axes for a 1 x 2 layout
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 6))

# First subplot: Fill between plot along y-axis (KDE Plot)
ax1.fill_betweenx(gdp_range, kde_early(gdp_range), color="lightgreen", alpha=0.5)
ax1.fill_betweenx(gdp_range, kde_mid(gdp_range), color="saddlebrown", alpha=0.5)
ax1.fill_betweenx(gdp_range, kde_recent(gdp_range), color="forestgreen", alpha=0.5)
ax1.plot(kde_early(gdp_range), gdp_range, label=labels[0], color="green", linestyle='--', linewidth=2)
ax1.plot(kde_mid(gdp_range), gdp_range, label=labels[1], color="brown", linestyle='-.', linewidth=2)
ax1.plot(kde_recent(gdp_range), gdp_range, label=labels[2], color="darkgreen", linestyle='-', linewidth=2)
ax1.legend(title=ax1_legend_title, loc="upper right")
ax1.set_xlabel(ax1_xlabel)
ax1.set_yticks([])  # Hiding y-axis ticks for clarity

# Second subplot: Box plot for the same datasets along y-axis
box = ax2.boxplot(
    [gdp_early, gdp_mid, gdp_recent], vert=True, patch_artist=True, medianprops={"color": "black"}
)
colors = ["lightgreen", "saddlebrown", "forestgreen"]  # Color matching with KDE plot
for patch, color in zip(box["boxes"], colors):
    patch.set_facecolor(color)

ax2.set_ylabel(ax2_ylabel)
ax2.set_xticks([])  # Hiding x-axis ticks for clarity
ax2.set_ylim(0, 5)  # Setting limits for y-axis to align with the KDE plot

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_114.pdf", bbox_inches="tight")
