

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate new transportation-related data
x = np.random.normal(60, 15, 100)  # Vehicle speed in mph
y = 50 - 0.3 * x + np.random.rand(100) * 5  # Fuel efficiency in mpg

# Define line data for visual representation
line_x = [40, 50, 60, 70, 80]
line_y = [38, 35, 30, 26, 22]
diagonal_line = [[30, 90], [20, 60]]

# Define plot parameters
scatters_name = "Speed vs Fuel Efficiency"
bin_edges_name = "Speed Bins"
calibration_curve_name = "Efficiency Trend"

line_label = "Efficiency Trend"
xlabel = "Vehicle Speed (mph)"
ylabel = "Fuel Efficiency (mpg)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
gs = fig.add_gridspec(
    2,
    1,
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)

# no labels on the histogram x-axis
ax_histx.tick_params(axis="x", labelbottom=False)

# the scatter plot with customized colors and markers
sc = ax.scatter(x, y, c=x, cmap="Blues", s=50, edgecolor="darkblue", alpha=0.7)

# add a line on ax
ax.plot(
    line_x,
    line_y,
    color="green",
    linewidth=2,
    marker="^",
    markersize=7,
    label=line_label,
)

# draw a diagonal line
ax.plot(diagonal_line[0], diagonal_line[1], color="purple", linestyle="--")

# now determine nice limits by hand:
binwidth = 5
xymax = max(np.max(np.abs(x)), np.max(np.abs(y)))
lim = (int(xymax / binwidth) + 1) * binwidth
bins = np.arange(30, lim + binwidth, binwidth)
ax_histx.hist(x, bins=bins, color="#a3d6b3", edgecolor="darkgreen")

# remove the y-axis labels
ax_histx.set_yticks([])

# remove the top, left, and right spines
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)

# set axis labels
ax.set_xlabel(xlabel, fontsize=12, fontweight='bold')
ax.set_ylabel(ylabel, fontsize=12, fontweight='bold')

# set ticks and limits
ax.set_xticks([30, 40, 50, 60, 70, 80, 90])
ax.set_yticks([20, 30, 40, 50, 60])
ax.set_xlim(30, 90)
ax.set_ylim(20, 60)

# draw bin edges
for i in bins:
    ax.axvline(i, color="gray", linewidth=0.5, linestyle="--", zorder=0, alpha=0.5)

# Create an axes on the top side of ax_heatmap_top for the colorbar.
ax_colorbar = fig.add_axes(
    [ax.get_position().width * 1.15, 0.1, 0.05, ax.get_position().y1 * 0.86]
)
cbar = plt.colorbar(sc, cax=ax_colorbar, orientation="vertical")

# Create legend elements
bin_edge_line = Line2D([0], [0], color="gray", linewidth=0.5, linestyle="--")
calibration_curve_line = Line2D(
    [0],
    [0],
    color="green",
    linewidth=2,
    marker="^",
    markersize=7,
)

# Update legend to include the new Line2D instances
ax.legend(
    handles=[sc, bin_edge_line, calibration_curve_line],
    labels=[scatters_name, bin_edges_name, calibration_curve_name],
    loc="upper left",
    fontsize=10
)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_38.pdf", bbox_inches="tight")
