

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate data related to weather - temperature and humidity
temperature = np.random.rand(100) * 50  # Temperature range between 0 and 50°C
humidity = 0.3 * temperature + np.random.rand(100) * 30  # Humidity in percentage

# Add a line for visualization purposes
line_x = [10, 20, 30, 40, 50]
line_y = [15, 25, 35, 45, 55]
diagonal_line = [[0, 50], [0, 50]]

scatters_name = "Temperature vs. Humidity"
bin_edges_name = "Bin Edges"
calibration_curve_name = "Trend Line"

line_label = "Trend Line"
xlabel = "Temperature (°C)"
ylabel = "Humidity (%)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Draw the scatter plot and marginals.

# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
gs = fig.add_gridspec(
    2,
    1,
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)

# no labels
ax_histx.tick_params(axis="x", labelbottom=False)

# the scatter plot:
sc = ax.scatter(temperature, humidity, c=temperature, cmap="Blues", s=70, edgecolor="black", alpha=0.75)

# add a line on ax
ax.plot(
    line_x,
    line_y,
    color="navy",
    linewidth=2,
    marker="o",
    markersize=8,
    label=line_label,
)

# draw a diagonal line
ax.plot(diagonal_line[0], diagonal_line[1], color="darkgrey", linestyle="--")

# now determine nice limits by hand:
binwidth = 5
xymax = max(np.max(np.abs(temperature)), np.max(np.abs(humidity)))
lim = (int(xymax / binwidth) + 1) * binwidth
bins = np.arange(0, lim + binwidth, binwidth)
ax_histx.hist(temperature, bins=bins, color="#87ceeb", edgecolor="black")

# remove the y-axis labels
ax_histx.set_yticks([])

# remove the top, left, and right spines
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)

ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

ax.set_xticks(np.arange(0, 55, 10))
ax.set_yticks(np.arange(0, 60, 10))
ax.set_xlim(-5, 55)
ax.set_ylim(-5, 60)

# draw bin edges
for i in bins:
    ax.axvline(i, color="gray", linewidth=0.5, linestyle="--", zorder=0, alpha=0.5)

# Create an axes on the top side of ax_heatmap_top for the colorbar.
ax_colorbar = fig.add_axes(
    [ax.get_position().width * 1.15, 0.1, 0.05, ax.get_position().y1 * 0.86]
)
cbar = plt.colorbar(sc, cax=ax_colorbar, orientation="vertical")

# Create a Line2D instance for bin edges
bin_edge_line = Line2D([0], [0], color="gray", linewidth=0.5, linestyle="--")

# Create a Line2D instance for calibration curve
calibration_curve_line = Line2D(
    [0],
    [0],
    color="navy",
    linewidth=2,
    marker="o",
    markersize=8,
)

# Update legend to include the new Line2D instances
ax.legend(
    handles=[sc, bin_edge_line, calibration_curve_line],
    labels=[scatters_name, bin_edges_name, calibration_curve_name],
    loc="upper left",
)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_39.pdf", bbox_inches="tight")

