

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Fixing random state for reproducibility
# Simulated psychology data
stress_levels = np.random.uniform(low=1, high=10, size=300)
sleep_hours = 10 - 0.3 * stress_levels + np.random.normal(loc=0, scale=1, size=300)

xlabel = "Stress Levels (1-10)"
ylabel = "Sleep Hours (hrs)"
title = "Scatter Plot of Stress Levels vs Sleep Hours"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
# Add a gridspec with two rows and two columns and a ratio of 1 to 4 between
# the size of the marginal axes and the main axes in both directions.
# Also adjust the subplot parameters for a square plot.
gs = fig.add_gridspec(
    2,
    2,
    width_ratios=(4, 1),
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)
ax_histy = fig.add_subplot(gs[1, 1], sharey=ax)

# Draw the scatter plot and marginals.
# No labels
ax_histx.tick_params(axis="x", labelbottom=False)
ax_histy.tick_params(axis="y", labelleft=False)

# The scatter plot:
ax.scatter(stress_levels, sleep_hours, color="#2ca02c", alpha=0.6, edgecolor="black", marker='o')

# Now determine nice limits by hand:
binwidth = 1
xymax = max(np.max(np.abs(stress_levels)), np.max(np.abs(sleep_hours)))
lim = (int(xymax / binwidth) + 1) * binwidth

bins = np.arange(0, lim + binwidth, binwidth)
ax_histx.hist(stress_levels, bins=bins, color="lightgreen", edgecolor="#2ca02c")
ax_histy.hist(
    sleep_hours, bins=bins, orientation="horizontal", color="lightgreen", edgecolor="#2ca02c"
)

# Remove the y-axis labels
ax_histx.set_yticks([])
ax_histy.set_xticks([])

# Add grid lines to the scatter plot for better readability
ax.grid(True, linestyle='--', linewidth=0.5)

# Set axis labels
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_title(title, fontsize=14, pad=20)

# Remove the top, left, and right spines of the histograms
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)
ax_histy.spines["top"].set_visible(False)
ax_histy.spines["bottom"].set_visible(False)
ax_histy.spines["right"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_43.pdf", bbox_inches="tight")
