
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Define parameters for three Gaussian distributions representing different subjects
distributions = {
    "Math": {"mean": [20, 75], "cov": [[10, 15], [15, 80]]},
    "Science": {"mean": [22, 70], "cov": [[8, 12], [12, 70]]},
    "Literature": {"mean": [18, 65], "cov": [[9, 10], [10, 65]]},
}
keys = ["Math", "Science", "Literature"]
# Generate samples
samples = {
    subject: np.random.multivariate_normal(dist["mean"], dist["cov"], 100)
    for subject, dist in distributions.items()
}

xlabel = "Study Hours per Week"
ylabel = "Exam Scores"
title = "Student Performance by Subject"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the plotting grid
fig = plt.figure(figsize=(10, 8))
grid = plt.GridSpec(4, 4, hspace=0, wspace=0)

# Main scatter plot
main_ax = fig.add_subplot(grid[1:, :-1])
colors = {keys[0]: "navy", keys[1]: "skyblue", keys[2]: "teal"}
markers = {keys[0]: "o", keys[1]: "s", keys[2]: "D"}
for subject, color in colors.items():
    subset = samples[subject]
    main_ax.scatter(subset[:, 0], subset[:, 1], c=color, label=subject, alpha=0.6, marker=markers[subject])
main_ax.set_xlabel(xlabel, fontsize=12)
main_ax.set_ylabel(ylabel, fontsize=12)
main_ax.grid(True, linestyle='--', alpha=0.7)

# Top density plot
top_ax = fig.add_subplot(grid[0, :-1], sharex=main_ax)
all_samples = np.concatenate([samples[subject] for subject in samples], axis=0)
x_min, x_max = all_samples[:, 0].min(), all_samples[:, 0].max()
xs = np.linspace(x_min, x_max, 200)
for subject, color in colors.items():
    density = gaussian_kde(samples[subject][:, 0])
    top_ax.plot(xs, density(xs), color=color, linestyle='--')

# Right density plot
right_ax = fig.add_subplot(grid[1:, -1], sharey=main_ax)
y_min, y_max = all_samples[:, 1].min(), all_samples[:, 1].max()
ys = np.linspace(y_min, y_max, 200)
for subject, color in colors.items():
    density = gaussian_kde(samples[subject][:, 1])
    right_ax.plot(density(ys), ys, color=color, linestyle='--')

# Hide the spines
top_ax.spines["top"].set_visible(False)
top_ax.spines["right"].set_visible(False)
top_ax.spines["left"].set_visible(False)
right_ax.spines["top"].set_visible(False)
right_ax.spines["right"].set_visible(False)
right_ax.spines["bottom"].set_visible(False)

# Remove the labels from the top and right axes
top_ax.tick_params(axis="x", which="both", top=False, bottom=False, labelbottom=False)
right_ax.tick_params(axis="y", which="both", left=False, right=False, labelleft=False)

# Remove all ticks from the right axis
top_ax.set_yticks([])
right_ax.set_xticks([])

main_ax.legend(title=title, fontsize=10)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_46.pdf", bbox_inches="tight")

