
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Generate random data to simulate box office revenues of two movie franchises
data_franchiseA = np.random.normal(500, 100, 50)  # FranchiseA box office revenue in millions
data_franchiseB = np.random.normal(700, 150, 50)  # FranchiseB box office revenue in millions

# Combine data into a list
data = [data_franchiseA, data_franchiseB]

# Create positions for each box plot
positions = [0, 1]
xticks = ["FranchiseA", "FranchiseB"]
xlabel = "Movie Franchises"
ylabel = "Box Office Revenue (USD Millions)"
title = "Box Office Revenue Distribution"
legend_labels = ["FranchiseA KDE", "FranchiseB KDE"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
plt.figure(figsize=(8, 6))

# Calculate the kernel density estimate and plot the density plot for each dataset
colors = ["#2ca02c", "#1f77b4"]  # Green for FranchiseA, Blue for FranchiseB
for i, d in enumerate(data):
    kde = gaussian_kde(d)
    kde_x = np.linspace(min(d), max(d), 300)
    kde_y = kde(kde_x)
    kde_y_scaled = kde_y / kde_y.max() * 0.1  # Scale factor can be adjusted
    offset = 0.2
    plt.fill_betweenx(
        kde_x,
        positions[i] - kde_y_scaled - offset,
        positions[i] - offset,
        color=colors[i],
        edgecolor="black",
        alpha=0.6,
        label=legend_labels[i]
    )

# Create box plots inside the violin plots
for i, d in enumerate(data):
    plt.boxplot(
        d,
        positions=[positions[i]],
        widths=0.15,
        patch_artist=True,
        medianprops=dict(color="black"),
        boxprops=dict(facecolor="none", color="black"),
    )

# Add scatter plot for individual data points
for i, d in enumerate(data):
    x = np.random.normal(positions[i], 0.04, size=len(d))
    plt.scatter(x, d, color=colors[i], s=10)

# Set the x-axis labels and add title
plt.xticks(positions, xticks)
plt.xlabel(xlabel)
plt.ylabel(ylabel)
plt.title(title)
plt.legend()

# Adjust the y-axis limits based on new data
plt.ylim(200, 1000)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_78.pdf", bbox_inches="tight")
