
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate synthetic data for an education study: Study Hours vs. Exam Scores
study_hours = np.random.rand(100) * 8 + 2  # Study hours between 2 and 10
exam_scores = 50 + study_hours * 5 + np.random.randn(100) * 5  # Scores between 50 and 100 with some noise

# Hypothetical trend line representing correlation
trend_x = [2, 4, 6, 8, 10]
trend_y = [60, 70, 80, 90, 100]

# Labels and titles
scatter_name = "Students"
line_name = "Trend Line"
xlabel = "Study Hours"
ylabel = "Exam Scores"
title = "Study Hours vs. Exam Scores Study"
legend_labels = ["Students", "Trend Line", "Reference Line"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
gs = fig.add_gridspec(
    2,
    1,
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)

# Draw the scatter plot and marginals.
ax_histx.tick_params(axis="x", labelbottom=False)

# Scatter plot
ax.scatter(study_hours, exam_scores, color="#4B9CD3", edgecolor="white", s=50, label=scatter_name)

# Trend line
ax.plot(trend_x, trend_y, color="#2E86C1", linewidth=2, marker="o", markersize=6, label=line_name)

# Reference diagonal line
ax.plot([2, 10], [50, 100], color="#85C1E9", linestyle="--")

# Histogram for the x-axis data (Study Hours)
binwidth = 0.2
xymax = max(np.max(np.abs(study_hours)), np.max(np.abs(exam_scores)))
lim = (int(xymax / binwidth) + 1) * binwidth
bins = np.arange(0, lim + binwidth, binwidth)
ax_histx.hist(study_hours, bins=bins, color="#AED6F1", edgecolor="black")

# Remove the y-axis labels
ax_histx.set_yticks([])

# Remove the top, left, and right spines
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)

# Labeling
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

ax.set_xticks([2, 4, 6, 8, 10])
ax.set_yticks([50, 60, 70, 80, 90, 100])
ax.set_xlim(1.5, 10.5)
ax.set_ylim(45, 105)

ax.legend(loc="upper left")

# Add grid for better readability
ax.grid(True)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_89.pdf", bbox_inches="tight")
