
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import norm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Define the data for each subplot using random numbers and normal distribution
wheat_yield = np.random.normal(8.0, 1.5, 1000)  # Wheat yield in tons per hectare
corn_yield = np.random.normal(10.0, 2.0, 1000)  # Corn yield in tons per hectare
rice_yield = np.random.normal(6.5, 1.3, 1000)  # Rice yield in tons per hectare
soybean_yield = np.random.normal(3.0, 0.8, 1000)  # Soybean yield in tons per hectare
titles = ["Wheat Yield", "Corn Yield", "Rice Yield", "Soybean Yield"]
xlabels = [
    "Yield (tons/ha)",
    "Yield (tons/ha)",
    "Yield (tons/ha)",
    "Yield (tons/ha)",
]
ylabels = ["Probability", "Probability", "Probability", "Probability"]
super_title = "Agricultural Yield Statistics"
annotation_positions = {
    "Wheat Yield": (0.6, 0.8),
    "Corn Yield": (0.6, 0.8),
    "Rice Yield": (0.6, 0.8),
    "Soybean Yield": (0.6, 0.8),
}

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size to match the original image's dimensions
plt.figure(figsize=(12, 10))

# Define the subplot grid with reduced space between subplots
grid = plt.GridSpec(2, 2, wspace=0.3, hspace=0.4)

# Color mapping for each dataset
colors = {
    "Wheat Yield": "#ff9999",
    "Corn Yield": "#66b3ff",
    "Rice Yield": "#99ff99",
    "Soybean Yield": "#ffcc99"
}

# Wheat yield subplot
ax_wheat = plt.subplot(grid[0, 0])
ax_wheat.hist(
    wheat_yield, bins=20, color=colors["Wheat Yield"], edgecolor="black", density=True, linewidth=1
)
ax_wheat.plot(
    np.sort(wheat_yield),
    norm.pdf(np.sort(wheat_yield), np.mean(wheat_yield), np.std(wheat_yield)),
    color="#555555",
    linestyle='--'
)
ax_wheat.set_title(titles[0])
ax_wheat.set_xlabel(xlabels[0])
ax_wheat.set_ylabel(ylabels[0])
ax_wheat.set_ylim(0, 0.3)
ax_wheat.text(
    annotation_positions["Wheat Yield"][0],
    annotation_positions["Wheat Yield"][1],
    f"$\mu = {np.mean(wheat_yield):.2f}$\n$\sigma = {np.std(wheat_yield):.2f}$",
    transform=ax_wheat.transAxes,
    fontsize=9,
)

# Corn yield subplot
ax_corn = plt.subplot(grid[0, 1])
ax_corn.hist(
    corn_yield,
    bins=20,
    color=colors["Corn Yield"],
    edgecolor="black",
    density=True,
    linewidth=1,
)
ax_corn.plot(
    np.sort(corn_yield),
    norm.pdf(np.sort(corn_yield), np.mean(corn_yield), np.std(corn_yield)),
    color="#555555",
    linestyle='--'
)
ax_corn.set_title(titles[1])
ax_corn.set_xlabel(xlabels[1])
ax_corn.set_ylabel(ylabels[1])
ax_corn.set_ylim(0, 0.25)
ax_corn.text(
    annotation_positions["Corn Yield"][0],
    annotation_positions["Corn Yield"][1],
    f"$\mu = {np.mean(corn_yield):.2f}$\n$\sigma = {np.std(corn_yield):.2f}$",
    transform=ax_corn.transAxes,
    fontsize=9,
)

# Rice yield subplot
ax_rice = plt.subplot(grid[1, 0])
ax_rice.hist(
    rice_yield,
    bins=20,
    color=colors["Rice Yield"],
    edgecolor="black",
    density=True,
    linewidth=1,
)
ax_rice.plot(
    np.sort(rice_yield),
    norm.pdf(np.sort(rice_yield), np.mean(rice_yield), np.std(rice_yield)),
    color="#555555",
    linestyle='--'
)
ax_rice.set_title(titles[2])
ax_rice.set_xlabel(xlabels[2])
ax_rice.set_ylabel(ylabels[2])
ax_rice.set_ylim(0, 0.35)
ax_rice.text(
    annotation_positions["Rice Yield"][0],
    annotation_positions["Rice Yield"][1],
    f"$\mu = {np.mean(rice_yield):.2f}$\n$\sigma = {np.std(rice_yield):.2f}$",
    transform=ax_rice.transAxes,
    fontsize=9,
)

# Soybean yield subplot
ax_soybean = plt.subplot(grid[1, 1])
ax_soybean.hist(
    soybean_yield, bins=20, color=colors["Soybean Yield"], edgecolor="black", density=True, linewidth=1
)
ax_soybean.plot(
    np.sort(soybean_yield),
    norm.pdf(np.sort(soybean_yield), np.mean(soybean_yield), np.std(soybean_yield)),
    color="#555555",
    linestyle='--'
)
ax_soybean.set_title(titles[3])
ax_soybean.set_xlabel(xlabels[3])
ax_soybean.set_ylabel(ylabels[3])
ax_soybean.set_ylim(0, 0.7)
ax_soybean.text(
    annotation_positions["Soybean Yield"][0],
    annotation_positions["Soybean Yield"][1],
    f"$\mu = {np.mean(soybean_yield):.2f}$\n$\sigma = {np.std(soybean_yield):.2f}$",
    transform=ax_soybean.transAxes,
    fontsize=9,
)

# Add a super title to the figure for context
plt.suptitle(super_title, fontsize=16)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout(rect=[0, 0.03, 1, 0.97])  # Adjust rect to accommodate the super title
plt.savefig("CB_95.pdf", bbox_inches="tight")
