
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Simulate time slots (hours of the day)
time_slots = np.linspace(0, 24, 100)
# Simulate energy consumption levels for different sectors
consumption_residential = 0.25 - 0.004 * (time_slots - 8) ** 2
consumption_commercial = 0.2 - 0.003 * (time_slots - 12) ** 2
consumption_industrial = 0.3 - 0.005 * (time_slots - 16) ** 2
consumption_transportation = 0.15 - 0.0025 * (time_slots - 20) ** 2
consumption_public_services = 0.1 - 0.002 * (time_slots - 10) ** 2

# Simulate energy usage preference matrix
Residential = np.random.rand(3, 10) * 15
Commercial = np.random.rand(2, 10) * 15
Industrial = np.random.rand(4, 10) * 15
Transportation = np.random.rand(3, 10) * 15
PublicServices = np.random.rand(2, 10) * 15

# Sort simulated consumption data
Residential = [sorted(r, reverse=True) for r in Residential]
Commercial = [sorted(c, reverse=True) for c in Commercial]
Industrial = [sorted(i, reverse=True) for i in Industrial]
Transportation = [sorted(t) for t in Transportation]
PublicServices = [sorted(ps) for ps in PublicServices]

# Label parameters
xlabel = "Time (Hours)"
ylabel = "Consumption Level"
title = "Energy Consumption Across Different Sectors"
baseline = 0
labels = ["Residential", "Commercial", "Industrial", "Transport", "Public Services"]
textheight = 9.5
xlabel2 = "Segments"
ylabel2 = "Time Slots"
title2 = "Energy Usage Preference Matrix"
plotlabels = ["Residential", "Commercial", "Industrial", "Transport", "Public Services"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
plt.figure(figsize=(10, 6))

# New color scheme for energy visualization
colors = ["#FF6F61", "#6B5B95", "#88B04B", "#F7CAC9", "#92A8D1"]

# Create the left plot (Energy Consumption in Different Sectors)
plt.subplot(1, 2, 1)
plt.plot(time_slots, consumption_residential, label=plotlabels[0], color=colors[0], linestyle='-', linewidth=2)
plt.plot(time_slots, consumption_commercial, label=plotlabels[1], color=colors[1], linestyle='--', linewidth=2)
plt.plot(time_slots, consumption_industrial, label=plotlabels[2], color=colors[2], linestyle='-.', linewidth=2)
plt.plot(time_slots, consumption_transportation, label=plotlabels[3], color=colors[3], linestyle=':', linewidth=2)
plt.plot(time_slots, consumption_public_services, label=plotlabels[4], color=colors[4], linestyle=(0, (3, 1, 1, 1)), linewidth=2)

plt.gca().spines["right"].set_visible(False)
plt.gca().spines["top"].set_visible(False)

plt.xlabel(xlabel)
plt.ylabel(ylabel)
plt.title(title)
plt.legend()

# Create the right plot (Energy usage preference matrix)
plt.subplot(1, 2, 2)

for index, values in enumerate([Residential, Commercial, Industrial, Transportation, PublicServices]):
    for i in range(len(values)):
        plt.scatter(
            [baseline + i] * len(values[i]),
            range(len(values[i])),
            s=values[i],
            c=colors[index],
            marker='o' if index % 2 == 0 else '^'
        )
    plt.text(baseline + len(values) / 2-1, textheight, labels[index], fontsize=8)
    baseline = baseline + len(values)

for spine in plt.gca().spines.values():
    spine.set_visible(False)

plt.xticks([])
plt.yticks([])
plt.xlabel(xlabel2)
plt.ylabel(ylabel2)
plt.title(title2, y=1.05)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and show plot
plt.tight_layout()
plt.savefig("HR_45.pdf", bbox_inches="tight")
