

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Seed for reproducibility
np.random.seed(42)

# Function to simulate monthly heart rate trends
def heart_rate_trends(x, n=3):
    """Return a random mixture of heart rate trends for *n* individuals, evaluated at positions *x*."""
    
    def add_random_trend(a):
        amplitude = 15 / (0.1 + np.random.random())
        dx = x[-1] - x[0]
        x0 = (0.5 + np.random.random()) * dx
        z = 3 / (0.1 + np.random.random()) / dx
        a += amplitude * np.exp(-((z * (x - x0)) ** 2))

    a = np.zeros_like(x)
    for j in range(n):
        add_random_trend(a)
    return 60 + a  # Baseline heart rate of 60 bpm with variations

# X-axis representing time
x = np.linspace(0, 12, 150)  # Months of the year

# Generating data for 3 different individuals
ys = [heart_rate_trends(x) for _ in range(3)]

# Labels, titles and other text
xlabel = "Time (Months)"
ylabel = "Heart Rate (BPM)"
title = "Monthly Heart Rate Trends"
legend_labels = ["Individual A", "Individual B", "Individual C"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Custom color scheme for health data
colors = ['#8BC34A', '#FFA726', '#FFEB3B']

# Plotting with updated styles and color scheme
ax.stackplot(x, ys, baseline="wiggle", colors=colors, linewidth=1)

# Adding labels and title
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)
ax.set_title(title, fontsize=16)

# Adding legend
ax.legend(legend_labels, loc='upper left', fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_76.pdf", bbox_inches="tight")
