
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42) # Different seed for new data

# Data for philosophy paper publications
xlabels = ["January", "February", "March", "April", "May"]
categoryA = np.array([15, 22, 30, 24, 29])
categoryB = np.array([17, 21, 19, 27, 23])

# Labels and Plot Types
hat_graph_label = ["Category A", "Category B"]

# Axes Limits and Labels
xlabel_value = "Months"
ylabel_value = "Number of Published Papers"
ylim_values = [0, 40]
title = "Philosophy Papers by Month for Each Category"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
def hat_graph(ax, xlabels, values, group_labels):
    def label_bars(heights, rects):
        """Attach a text label on top of each bar."""
        for height, rect in zip(heights, rects):
            ax.annotate(
                f"{height}",
                xy=(rect.get_x() + rect.get_width() / 2, height),
                xytext=(0, 4),  # 4 points vertical offset.
                textcoords="offset points",
                ha="center",
                va="bottom",
            )

    values = np.asarray(values)
    x = np.arange(values.shape[1])
    ax.set_xticks(x, labels=xlabels)
    spacing = 0.3  # spacing between hat groups
    width = (1 - spacing) / values.shape[0]
    heights0 = values[0]
    for i, (heights, group_label) in enumerate(zip(values, group_labels)):
        style = (
            {"fill": False, "edgecolor": "brown", "hatch": "//"}
            if i == 0
            else {"edgecolor": "darkblue", "color": "#ffcc00"}
        )
        rects = ax.bar(
            x - spacing / 2 + i * width,
            heights - heights0,
            width,
            bottom=heights0,
            label=group_label,
            **style,
        )
        label_bars(heights, rects)

fig, ax = plt.subplots(figsize=(10, 6))
hat_graph(ax, xlabels, [categoryA, categoryB], hat_graph_label)

# Add some text for labels, title and custom x-axis tick labels, etc.
ax.set_xlabel(xlabel_value)
ax.set_ylabel(ylabel_value)
ax.set_ylim(ylim_values)
ax.set_title(title)
ax.legend()

# Optimize style further with grid and backdrop
ax.grid(True, linestyle='--', alpha=0.6)

# ===================
# Part 4: Saving Output
# ===================
fig.tight_layout()
plt.savefig("HR_81.pdf", bbox_inches="tight")
