
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data
categories = ["Solar", "Wind", "Hydro", "Geothermal"]
models = [
    "North America",
    "Europe",
    "Asia",
    "South America",
    "Africa",
]
values = np.array([
    [0.7, 0.65, 0.6, 0.55, 0.5],
    [0.5, 0.55, 0.6, 0.45, 0.4],
    [0.4, 0.45, 0.5, 0.35, 0.3],
    [0.3, 0.35, 0.4, 0.25, 0.2]
])
referlines = [0.6, 0.45, 0.35, 0.25]
ylabel = "Average Energy Output (MW)"
ylim = [0, 1]
arrowstart = (0.05, 0.25)
arrowend = (0.48, 0.25)
arrowstart2 = (0.55, 0.25)
arrowend2 = (0.9, 0.25)
xlim = [-0.5, 3.5]
textposition = [[0.5, 1.2], [2.5, 1.2]]
textlabel = "Energy Source"
spanposition = [[-0.5, 1.5], [1.5, 3.5]]
xlabel = "Energy Source Categories"
title = "Average Energy Output by Source in Different Regions"
legendlabels = models

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
colors = ["#ff9999", "#66b3ff", "#99ff99", "#ffcc99", "#c44e52"]
# Plotting
fig, ax = plt.subplots(figsize=(10, 5))
width = 0.1
x = np.arange(len(categories))
ax.axvspan(
    spanposition[0][0],
    spanposition[0][1],
    color="#e6f7ff",
)
ax.axvspan(
    spanposition[1][0],
    spanposition[1][1],
    color="#ffebf0",
)

for i, subcategory in enumerate(categories):
    for j, (model, color, referline) in enumerate(zip(models, colors, referlines)):
        ax.bar(
            i + (j - 2) * width,
            values[i, j] - referline,
            width,
            bottom=referline,
            label=model if i == 0 else "",
            color=color,
        )

# Annotations
for k, model in enumerate(models):
    for i, category in enumerate(categories):
        ax.text(
            i + (k - 2) * width, values[i, k] + 0.02,f"{values[i, k]:.2f}",
            ha="center",
            fontdict={'fontsize':8}
        )

for line, color in zip(referlines, colors):
    ax.axhline(line, color=color, linestyle="--")
    ax.text(3.4, line + 0.04, f"{line:.2f}", ha="center", color=color)

# Customizations
ax.set_ylabel(ylabel)
ax.set_xticks(x)
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.set_xlim(xlim)
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.set_xlabel(xlabel)
plt.title(title)
plt.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    xycoords="figure fraction",
    arrowprops=dict(arrowstyle="<->", color="green", lw=1),
)
plt.annotate(
    "",
    xy=arrowstart2,
    xytext=arrowend2,
    xycoords="figure fraction",
    arrowprops=dict(arrowstyle="<->", color="purple", lw=1),
)
current_ticks = ax.get_xticks()
new_ticks = current_ticks + 0.5
# Set the new ticks without labels
ax.set_xticks(new_ticks, minor=True)  # Add as minor ticks
ax.xaxis.set_minor_formatter(plt.NullFormatter())  # Hide labels for minor ticks

# Enable grid for minor ticks, adjust grid appearance as needed
ax.grid(which="minor", color="black", linestyle="--", alpha=0.5)
ax.text(
    textposition[0][0],
    textposition[0][1]-0.3,
    textlabel,
    ha="center",
    va="top",
    fontsize=12,
    color="green",
)
ax.text(
    textposition[1][0],
    textposition[1][1]-0.3,
    textlabel,
    ha="center",
    va="top",
    fontsize=12,
    color="purple",
)
ax.legend(ncol=5, loc="upper center", bbox_to_anchor=(0.5, 1), labels=legendlabels)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_85.pdf", bbox_inches="tight")
