
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generating new entertainment data
clusters = {
    "movies": np.random.normal(loc=(0, 2), scale=1, size=(50, 2)),
    "music": np.random.normal(loc=(5, 5), scale=1, size=(30, 2)),
    "tv_shows": np.random.normal(loc=(-3, 0), scale=1, size=(40, 2)),
    "video_games": np.random.normal(loc=(5, -5), scale=1, size=(60, 2)),
    "sports": np.random.normal(loc=(-5, -5), scale=1, size=(70, 2)),
    "books": np.random.normal(loc=(5, 10), scale=1, size=(45, 2)),
}

insetaxes = [0.2, 0.6, 0.3, 0.2]    # Inset position
insetxlim = [-4, -2]                # Inset x-axis limit relevant to 'tv_shows'
insetylim = [-1, 1]                 # Inset y-axis limit relevant to 'tv_shows'
insetxticks = [-4.0, -3.0, -2.0]    # Inset x-axis ticks
insetyticks = [-1.0, 0, 1.0]        # Inset y-axis ticks
arrowstart = (-3.0, 4.0)            # Arrow start coordinates
arrowend = (0.3, 0.5)              # Arrow end coordinates
annotaterecx = [-4, -2]             # Annotation box x-coordinates around 'tv_shows'
annotaterecy = [-1, 1]              # Annotation box y-coordinates around 'tv_shows'
xlabel = "Popularity Score"         # x-axis label
ylabel = "Consumer Ratings"         # y-axis label
title = "Entertainment Categories Clusters" # Plot title
inset_title = "Zoomed View of TV Shows"     # Inset title

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the scatter plot

# Colors for each cluster reflecting entertainment categories
colors = {
    "movies": "#636363",       # grey
    "music": "#31a354",        # green
    "tv_shows": "#ffcc00",     # yellow
    "video_games": "#d95f0e",  # brown
    "sports": "#1f78b4",       # blue
    "books": "#e31a1c",        # red
}

# Marker shapes for each cluster
markers = {
    "movies": "o",
    "music": "s",
    "tv_shows": "D",
    "video_games": "^",
    "sports": "P",
    "books": "X",
}

fig, ax = plt.subplots(figsize=(8, 8))
for cluster, data in clusters.items():
    ax.scatter(data[:, 0], data[:, 1], c=colors[cluster], marker=markers[cluster], alpha=0.7, label=cluster)

# Draw annotation rectangle with dashed lines
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[1], annotaterecy[1]], color="black", lw=1, linestyle='--')
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[0], annotaterecy[0]], color="black", lw=1, linestyle='--')
ax.plot([annotaterecx[0], annotaterecx[0]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1, linestyle='--')
ax.plot([annotaterecx[1], annotaterecx[1]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1, linestyle='--')

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(insetaxes)  # Adjust the position to align with the right side of the main plot
for cluster, data in clusters.items():
    ax_inset.scatter(data[:, 0], data[:, 1], c=colors[cluster], marker=markers[cluster], alpha=0.7)
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")
ax_inset.set_title(inset_title)

ax.annotate("", xy=arrowstart, xytext=arrowend, textcoords="axes fraction", arrowprops=dict(facecolor="black", lw=0.1))

# Set labels and title
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)
ax.legend(ncol=3)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_45.pdf", bbox_inches="tight")
