
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate Years
years = np.linspace(2000, 2050, 50)

# Generate different CO2 emissions trends for each continent
continent1_emissions = np.exp((years - 2000) / 50) * 5  # Exponential increase
continent2_emissions = np.piecewise(years, [years < 2030, years >= 2030], 
                                    [lambda x: 10 + (x - 2000) * 0.5, 
                                     lambda x: 25 + np.random.normal(0, 1.0, len(x))])  # Plateau after 2030
continent3_emissions = np.random.normal(loc=15, scale=2.0, size=len(years))  # Relatively stable with fluctuations

# Simulate standard deviations for error
std_dev = 1.0
continent1_std = np.full_like(continent1_emissions, std_dev)
continent2_std = np.full_like(continent2_emissions, std_dev)
continent3_std = np.full_like(continent3_emissions, std_dev)

# Text parameters
xlabel = "Year"
ylabel = "CO2 Emissions (MtCO2)"
labels = ["Continent 1 (Exponential Increase)", "Continent 2 (Gradual Plateau)", "Continent 3 (Stable)"]
title = "CO2 Emissions Trends Over Time in Different Continents"
suptitle = "Environmental Science Data"
legendlabels = ["Continent 1", "Continent 2", "Continent 3"]
arrowlabel = "Gradual Plateau"
xticks = np.linspace(2000, 2050, 11)
yticks = np.arange(0, 60, 10)
axesinset = [0.7, 0.15, 0.2, 0.2]
insetxlim = [2025, 2040]
insetylim = [20, 30]
insetxticks = [2025, 2030, 2035, 2040]
insetyticks = [20, 25, 30]
arrowstart = (2030, 30)
arrowend = (0.365, 1.22)
annotaterecx = [2028, 2042]
annotaterecy = [20, 27]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a single plot
fig, ax = plt.subplots(figsize=(8, 6))

# Plot the first trend
ax.plot(years, continent1_emissions, "o--", color="forestgreen", label=labels[0], linewidth=1.5)
ax.fill_between(years, continent1_emissions - continent1_std, continent1_emissions + continent1_std, color="forestgreen", alpha=0.2)

# Plot the second trend
ax.plot(years, continent2_emissions, "s-", color="darkorange", label=labels[1], linewidth=1.5)
ax.fill_between(years, continent2_emissions - continent2_std, continent2_emissions + continent2_std, color="darkorange", alpha=0.2)

# Plot the third trend
ax.plot(years, continent3_emissions, "d-.", color="royalblue", label=labels[2], linewidth=1.5)
ax.fill_between(years, continent3_emissions - continent3_std, continent3_emissions + continent3_std, color="royalblue", alpha=0.2)

# Set labels, ticks, title, legend, and grid for the main plot
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_xticks(xticks)
ax.set_yticks(yticks)
ax.set_title(title, fontsize=14)
fig.suptitle(suptitle, fontsize=16)
ax.legend(loc="upper left", shadow=True, frameon=True, framealpha=0.9)
ax.grid(True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5)
ax.set_facecolor("#f0f0f0")

# Draw a rectangle on the main plot to indicate the area of zoom-in
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[1], annotaterecy[1]], color="black", lw=1)
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[0], annotaterecy[0]], color="black", lw=1)
ax.plot([annotaterecx[0], annotaterecx[0]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1)
ax.plot([annotaterecx[1], annotaterecx[1]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(axesinset)

# Plot the first trend on the inset
ax_inset.plot(years, continent1_emissions, "o--", color="forestgreen", label=labels[0], linewidth=1.5)
ax_inset.fill_between(years, continent1_emissions - continent1_std, continent1_emissions + continent1_std, color="forestgreen", alpha=0.2)

# Plot the second trend on the inset
ax_inset.plot(years, continent2_emissions, "s-", color="darkorange", label=labels[1], linewidth=1.5)
ax_inset.fill_between(years, continent2_emissions - continent2_std, continent2_emissions + continent2_std, color="darkorange", alpha=0.2)

# Plot the third trend on the inset
ax_inset.plot(years, continent3_emissions, "d-.", color="royalblue", label=labels[2], linewidth=1.5)
ax_inset.fill_between(years, continent3_emissions - continent3_std, continent3_emissions + continent3_std, color="royalblue", alpha=0.2)

# Set limits, ticks, and border color for the inset
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add an arrow from the rectangle on the main plot to the inset
ax.annotate("", xy=arrowstart, xytext=arrowend, textcoords="axes fraction", arrowprops=dict(facecolor="black", lw=0.5))

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("PIP_49.pdf", bbox_inches="tight")
