

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Generate days
days = np.linspace(0, 100, 50)

# Generate trends for politics
public_approval_rating = 50 + 10 * np.sin(days * 0.1)  # Sinusoidal trend (public approval rating)
media_coverage = 20 + (days * 0.5)  # Linear increase (media coverage)
approval_variation = public_approval_rating + np.random.normal(loc=0, scale=3, size=len(days))  # Adding noise
coverage_variation = media_coverage + np.random.normal(loc=0, scale=4, size=len(days))  # Adding noise 

# Simulate standard deviations for error
std_dev_approval = 3
std_dev_coverage = 4
public_approval_std = np.full_like(public_approval_rating, std_dev_approval)
media_coverage_std = np.full_like(media_coverage, std_dev_coverage)

# Labels and configuration
xlabel = "Days"
ylabel = "Political Metrics"
line_labels = ["Public Approval Rating", "Media Coverage"]
xticks = np.linspace(0, 100, 11)
yticks = np.linspace(0, 100, 11)
axesinset = [0.4, 0.75, 0.3, 0.2]
insetxlim = [40, 75]
insetylim = [30, 70]
insetxticks = [25, 50, 75]
insetyticks = [30, 50, 70]
arrowstart = (50, 75)
arrowend = (0.7, 0.35)
annotaterecx = [25, 75]
annotaterecy = [30, 70]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a single plot
fig, ax = plt.subplots(figsize=(8, 6))

# Plot the public approval rating trend on the main plot
ax.plot(days, approval_variation, 'o-', color='dodgerblue', label=line_labels[0])
ax.fill_between(
    days,
    approval_variation - public_approval_std,
    approval_variation + public_approval_std,
    color='lightblue',
    alpha=0.2,
)

# Plot the media coverage trend on the main plot
ax.plot(days, coverage_variation, 's-', color='forestgreen', label=line_labels[1])
ax.fill_between(
    days,
    coverage_variation - media_coverage_std,
    coverage_variation + media_coverage_std,
    color='lightgreen',
    alpha=0.2,
)

# Set labels, ticks, legend and grid for the main plot
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_xticks(xticks)
ax.set_yticks(yticks)
ax.set_ylim(0,100)
ax.legend(loc="upper left", shadow=True, frameon=True, framealpha=0.9)
ax.grid(
    True, which="both", axis="both", color="gray", linestyle="--", linewidth=0.5
)
ax.set_facecolor("#e8f4f8")

# Draw a rectangle on the main plot to indicate the area of zoom-in
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(
    axesinset
)  # Adjust the position to align with the right side of the main plot

# Plot the approval rating trend on the inset
ax_inset.plot(days, approval_variation, 'o-', color='dodgerblue', label=line_labels[0])
ax_inset.fill_between(
    days,
    approval_variation - public_approval_std,
    approval_variation + public_approval_std,
    color='lightblue',
    alpha=0.2,
)

# Plot the media coverage trend on the inset
ax_inset.plot(days, coverage_variation, 's-', color='forestgreen', label=line_labels[1])
ax_inset.fill_between(
    days,
    coverage_variation - media_coverage_std,
    coverage_variation + media_coverage_std,
    color='lightgreen',
    alpha=0.2,
)

# Set limits, ticks and border color for the inset
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("PIP_51.pdf", bbox_inches="tight")

