
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Updated days
days = np.linspace(0, 100, 50)

# Generate different trends for each philosophy metric
daily_reading_time = np.random.normal(loc=60, scale=10, size=len(days))  # Mean reading time in minutes
papers_published = np.cumsum(np.random.poisson(lam=0.5, size=len(days)))  # Cumulative count of papers published
conference_attendance = 50 + 30 * np.sin(days * 0.2)  # Sinusoidal pattern for conference attendance
new_concepts_introduced = np.linspace(0, 200, len(days)) + np.random.normal(loc=0, scale=20, size=len(days))  # Increasing trend for concepts

# Simulate standard deviations for error
std_dev = 10
daily_reading_time_std = np.full_like(daily_reading_time, std_dev)
papers_published_std = np.full_like(papers_published, std_dev / 2)
conference_attendance_std = np.full_like(conference_attendance, std_dev / 2)
new_concepts_introduced_std = np.full_like(new_concepts_introduced, std_dev)

# Labels and configuration
xlabel = "Days"
ylabel = "Philosophy Metrics"
line_labels = ["Daily Reading Time", "New Concepts Introduced"]
xticks = np.linspace(0, 100, 6)
yticks = np.arange(0, 300, 50)
axesinset = [0.2, 0.65, 0.3, 0.2]
insetxlim = [25, 75]
insetylim = [75, 150]
insetxticks = [25, 50, 75]
insetyticks = [75, 100, 125, 150]
arrowstart = (50, 120)
arrowend = (0.26, 0.6)
annotaterecx = [25, 75]
annotaterecy = [75, 150]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a single plot
fig, ax = plt.subplots(figsize=(6, 6))

# Plot the first line on the main plot
ax.plot(days, daily_reading_time, "o-", color="green", label=line_labels[0])
ax.fill_between(
    days,
    daily_reading_time - daily_reading_time_std,
    daily_reading_time + daily_reading_time_std,
    color="lightgreen",
    alpha=0.3,
)

# Plot the second line on the main plot
ax.plot(days, new_concepts_introduced, "s--", color="purple", label=line_labels[1])
ax.fill_between(
    days,
    new_concepts_introduced - new_concepts_introduced_std,
    new_concepts_introduced + new_concepts_introduced_std,
    color="lavender",
    alpha=0.3,
)

# Set labels, ticks, legend, and grid for the main plot
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_xticks(xticks)
ax.set_yticks(yticks)
ax.legend(loc="upper left", shadow=True, frameon=True, framealpha=0.9)
ax.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax.set_facecolor("#f9f9f9")

# Draw a rectangle on the main plot to indicate the area of zoom-in
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(
    axesinset
)  # Adjust the position to align with the right side of the main plot

# Plot the first line on the inset
ax_inset.plot(days, daily_reading_time, "o-", color="green", label=line_labels[0])
ax_inset.fill_between(
    days,
    daily_reading_time - daily_reading_time_std,
    daily_reading_time + daily_reading_time_std,
    color="lightgreen",
    alpha=0.3,
)

# Plot the second line on the inset
ax_inset.plot(days, new_concepts_introduced, "s--", color="purple", label=line_labels[1])
ax_inset.fill_between(
    days,
    new_concepts_introduced - new_concepts_introduced_std,
    new_concepts_introduced + new_concepts_introduced_std,
    color="lavender",
    alpha=0.3,
)

# Set limits, ticks, and border color for the inset
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add an arrow from the rectangle on the main plot to the inset
ax.annotate(
    "",
    xy=arrowstart,  # Arrow start point (on the main plot)
    xytext=arrowend,  # Arrow end point (on the inset)
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.1),
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("PIP_52.pdf", bbox_inches="tight")
