
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Simulated Data for Philosophy Trends
years = np.array([2010, 2012, 2014, 2016, 2018, 2020])
published_books = np.array([1500, 1800, 2100, 2500, 3000, 3700])  # Increasing trend for published philosophy books
academic_citations = np.array([3000, 3500, 4200, 5000, 6000, 7500])  # Increasing trend for academic citations

# Styling
labels = [
    "Published Philosophy Books",
    "Academic Citations in Philosophy"
]
xlabel = "Year"
ylabel = "Count"
title = "Published Philosophy Books and Academic Citations (2010-2020)"

# Labels and tick mark settings
xticks = years
yticks = np.linspace(0, 8000, 6)
yticklabels = [f"{int(x):,}" for x in yticks]
inset_axes = [0.2, 0.59, 0.25, 0.25]
inset_ylim = [1600, 4200]
yticks_inset = np.linspace(1600, 4200, 4)
x_years = [f"{int(x)}" for x in years]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the main figure and axis
fig, ax = plt.subplots(figsize=(10, 6))

# Plot the data with different styles and colors
ax.plot(years, published_books, "o-", label=labels[0], color="#FF5733", linewidth=2, markersize=8)
ax.plot(years, academic_citations, "x--", label=labels[1], color="#C70039", linewidth=2, markersize=8)

# Set labels and title
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_title(title, fontsize=14, fontweight='bold')

# Adjust y-axis limits and ticks
ax.set_ylim([0, 8000])
ax.set_yticks(yticks)
ax.set_yticklabels(yticklabels, fontsize=10)

# Add a legend
ax.legend(loc="upper left", fontsize=10)

# Create an inset axis for academic citations
ax_inset = fig.add_axes(inset_axes)
ax_inset.plot(years, academic_citations, "x--", color="#C70039", linewidth=2, markersize=8)

# Adjust y-axis limits for inset
ax_inset.set_ylim(inset_ylim)
ax_inset.set_xlim([2010, 2014])
ax_inset.set_yticks(yticks_inset)

# Change x-axis tick labels to years
ax.set_xticks(years)
ax.set_xticklabels(x_years, fontsize=10)
ax_inset.set_xticks([2010, 2012, 2014])
ax_inset.set_xticklabels(["2010", "2012", "2014"], fontsize=8)

# --------------------
# Part 4: Saving Output
# --------------------
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("PIP_71.pdf", bbox_inches="tight")
