
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data for transportation modes
import numpy as np

np.random.seed(42)
transport_modes = {
    "Car": np.random.normal(loc=(120, 15), scale=5, size=(50, 2)),
    "Bus": np.random.normal(loc=(80, 5), scale=5, size=(30, 2)),
    "Train": np.random.normal(loc=(100, 25), scale=5, size=(40, 2)),
    "Bicycle": np.random.normal(loc=(20, 0), scale=2, size=(60, 2)),
    "Motorbike": np.random.normal(loc=(90, 20), scale=5, size=(70, 2)),
    "Truck": np.random.normal(loc=(70, 3), scale=5, size=(45, 2)),
}

title = "Transportation Modes Performance Metrics"
xlabel = "Average Speed (km/h)"
ylabel = "Fuel Efficiency (km per liter)"
# Inset plot configuration
insetaxes = [0.1, 0.4, 0.3, 0.3]
insetxlim = [60, 100]  # Focus on Bus, Train, and Motorbike
insetylim = [0, 30]
insetxticks = [60.0, 80.0, 100.0]
insetyticks = [0.0, 15.0, 30.0]

# Arrow and annotation configuration
arrowstart = (70, 20)  # Arrow start for Motorbike
arrowend = (0.38, 0.5)  # Relative inset arrow end
annotaterecx = [60, 100]  # Annotation range for X
annotaterecy = [0, 30]  # Annotation range for Y

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Colors for transportation modes
colors = {
    "Car": "#1f77b4",
    "Bus": "#ff7f0e",
    "Train": "#2ca02c",
    "Bicycle": "#d62728",
    "Motorbike": "#9467bd",
    "Truck": "#8c564b",
}

# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 8))
for mode, data in transport_modes.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[mode],
        label=mode,
        alpha=0.6,
        edgecolors="w",
        s=100,
    )

# Enclosing rectangle for annotation
rect = plt.Rectangle(
    (annotaterecx[0], annotaterecy[0]),
    annotaterecx[1] - annotaterecx[0],
    annotaterecy[1] - annotaterecy[0],
    linewidth=1,
    edgecolor="black",
    facecolor="none",
)
ax.add_patch(rect)

# Create the inset with zoomed-in view
ax_inset = fig.add_axes(insetaxes)
for mode, data in transport_modes.items():
    ax_inset.scatter(
        data[:, 0], data[:, 1], c=colors[mode], alpha=0.6, edgecolors="w", s=100
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=1),
)

# Adding titles and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_75.pdf", bbox_inches="tight")
