
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
campaign_data = {
    "Candidate A": np.random.normal(loc=(500, 60), scale=10, size=(50, 2)),
    "Candidate B": np.random.normal(loc=(700, 70), scale=15, size=(30, 2)),
    "Candidate C": np.random.normal(loc=(300, 50), scale=12, size=(40, 2)),
    "Candidate D": np.random.normal(loc=(800, 80), scale=20, size=(60, 2)),
    "Candidate E": np.random.normal(loc=(400, 55), scale=10, size=(70, 2)),
    "Candidate F": np.random.normal(loc=(600, 65), scale=15, size=(45, 2)),
}

title = "Campaign Budget vs. Voter Approval Rating of Various Candidates"
xlabel = "Campaign Budget (in Thousands of Dollars)"
ylabel = "Voter Approval Rating (%)"
# Inset plot configuration
insetaxes = [0.3, 0.6, 0.3, 0.3]
insetxlim = [300, 700]  # Focus on the middle budget candidates
insetylim = [50, 80]
insetxticks = [400, 500, 600]
insetyticks = [55, 60, 65, 70, 75]

# Arrow and annotation configuration
arrowstart = (600, 65)  # Arrow start for focused area
arrowend = (0.30, 0.7)  # Relative inset arrow end
annotaterecx = [500, 700]  # Annotation range for X
annotaterecy = [60, 80]  # Annotation range for Y

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Colors for candidates
colors = {
    "Candidate A": "#ff9999",
    "Candidate B": "#66b3ff",
    "Candidate C": "#99ff99",
    "Candidate D": "#ffcc99",
    "Candidate E": "#c2c2f0",
    "Candidate F": "#ffb3e6",
}

# Markers for candidates
markers = {
    "Candidate A": "o",
    "Candidate B": "s",
    "Candidate C": "^",
    "Candidate D": "D",
    "Candidate E": "P",
    "Candidate F": "X",
}

# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 8))
for candidate, data in campaign_data.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[candidate],
        label=candidate,
        alpha=0.6,
        edgecolors="w",
        s=100,
        marker=markers[candidate],
    )

# Enclosing rectangle for annotation
rect = plt.Rectangle(
    (annotaterecx[0], annotaterecy[0]),
    annotaterecx[1] - annotaterecx[0],
    annotaterecy[1] - annotaterecy[0],
    linewidth=1,
    edgecolor="black",
    facecolor="none",
)
ax.add_patch(rect)

# Create the inset with zoomed-in view
ax_inset = fig.add_axes(insetaxes)
for candidate, data in campaign_data.items():
    ax_inset.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[candidate],
        alpha=0.6,
        edgecolors="w",
        s=40,
        marker=markers[candidate],
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=1),
)

# Adding titles and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_76.pdf", bbox_inches="tight")
