
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Simulated communication data (average daily usage time vs frequency)
communication_methods = {
    "email": np.random.normal(loc=(30, 10), scale=5, size=(50, 2)),  # Email
    "social_media": np.random.normal(loc=(120, 25), scale=10, size=(30, 2)),  # Social Media
    "phone_calls": np.random.normal(loc=(20, 8), scale=5, size=(40, 2)),  # Phone Calls
    "messaging": np.random.normal(loc=(60, 50), scale=10, size=(60, 2)),  # Messaging
    "video_calls": np.random.normal(loc=(45, 5), scale=5, size=(70, 2)),  # Video Calls
}

insetaxes = [0.5, 0.2, 0.2, 0.2]
insetxlim = [0, 75]
insetylim = [30, 60]
insetxticks = [0, 75]
insetyticks = [30, 60]
arrowstart = (30, 50)  # Messaging
arrowend = (0.5, 0.35)
annotaterecx = [0, 75]
annotaterecy = [30, 60]
title = "Communication Methods Analysis"
xlabel = "Average Daily Usage Time (minutes)"
ylabel = "Usage Frequency (times per day)"
legendlabels = ["Email", "Social Media", "Phone Calls", "Messaging", "Video Calls"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 10))

# Define colors and markers for each communication method
colors = {
    "email": "#FF5733",  # Email
    "social_media": "#33FF57",  # Social Media
    "phone_calls": "#3357FF",  # Phone Calls
    "messaging": "#FF33B5",  # Messaging
    "video_calls": "#FFC300",  # Video Calls
}

markers = {
    "email": "o",  # Email
    "social_media": "s",  # Social Media
    "phone_calls": "^",  # Phone Calls
    "messaging": "d",  # Messaging
    "video_calls": "p",  # Video Calls
}

for method, data in communication_methods.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[method],
        marker=markers[method],
        label=method,
        alpha=0.7,
    )

# Draw annotation rectangle around specific region
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Create the inset with a zoomed-in view of a specific region
ax_inset = fig.add_axes(insetaxes)
for method, data in communication_methods.items():
    ax_inset.scatter(
        data[:, 0], data[:, 1], c=colors[method], marker=markers[method], alpha=0.7
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add annotation to indicate the inset region
ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.1),
)

# Add titles and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.legend(legendlabels)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("PIP_79.pdf", bbox_inches="tight")
