
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data for the plot with new trends in the psychology domain
time_points = np.array([0, 5, 10, 15, 20])
cognitive_therapy = np.array([0.50, 0.55, 0.60, 0.65, 0.70]) + np.random.normal(0, 0.02, 5)
behavioral_therapy = np.array([0.45, 0.50, 0.55, 0.60, 0.65]) + np.random.normal(0, 0.015, 5)
humanistic_therapy = np.array([0.40, 0.45, 0.50, 0.55, 0.60]) + np.random.normal(0, 0.02, 5)
psychoanalysis = np.array([0.55, 0.58, 0.61, 0.64, 0.67]) + np.random.normal(0, 0.018, 5)
control_group = np.linspace(0.50, 0.50, len(time_points))

# Extracted variables
fill_label_cognitive_therapy = "Cognitive Therapy"
fill_label_behavioral_therapy = "Behavioral Therapy"
fill_label_humanistic_therapy = "Humanistic Therapy"
fill_label_psychoanalysis = "Psychoanalysis"
plot_label_control = "Control Group"
title_text = "Impact of Different Therapies on Psychological Well-Being Over Time"
xlabel_text = "Time (Months)"
ylabel_text = "Psychological Well-Being"
legend_title = "Therapies"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the plot with a different visualization style
plt.figure(figsize=(10, 6))

xlim_values = (min(time_points), max(time_points))
ylim_values = (0.35, 0.75)
xticks_values = time_points
yticks_values = [0.40, 0.45, 0.50, 0.55, 0.60, 0.65, 0.70, 0.75]
legend_loc = "upper left"
legend_bbox_to_anchor = (1, 1)
legend_ncol = 1

fill_colors = {
    "Cognitive Therapy": "#FF9999",
    "Behavioral Therapy": "#99CC99",
    "Humanistic Therapy": "#FFCC99",
    "Psychoanalysis": "#99CCFF"
}

plt.fill_between(
    time_points, cognitive_therapy, color=fill_colors["Cognitive Therapy"], alpha=0.5, label=fill_label_cognitive_therapy
)
plt.fill_between(
    time_points, behavioral_therapy, color=fill_colors["Behavioral Therapy"], alpha=0.5, label=fill_label_behavioral_therapy
)
plt.fill_between(
    time_points, humanistic_therapy, color=fill_colors["Humanistic Therapy"], alpha=0.5, label=fill_label_humanistic_therapy
)
plt.fill_between(
    time_points, psychoanalysis, color=fill_colors["Psychoanalysis"], alpha=0.5, label=fill_label_psychoanalysis
)
plt.plot(
    time_points, control_group, color="black", linestyle="--", linewidth=2, label=plot_label_control
)

# Add a title and labels with enhanced formatting
plt.title(title_text, fontsize=14, y=1.1)
plt.xlabel(xlabel_text, fontsize=12)
plt.ylabel(ylabel_text, fontsize=12)
plt.xticks(xticks_values)
plt.yticks(yticks_values)
plt.tick_params(axis="both", which="both", length=0)

# Setting the limits explicitly to prevent cut-offs
plt.xlim(*xlim_values)
plt.ylim(*ylim_values)

# Adding a legend with a title
plt.legend(
    title=legend_title,
    frameon=True,
    framealpha=0.8,
    loc=legend_loc,
    bbox_to_anchor=legend_bbox_to_anchor,
    ncol=legend_ncol,
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to ensure no clipping
plt.tight_layout()
plt.savefig("area_29.pdf", bbox_inches="tight")
