
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Data for the plot representing renewable energy adoption over years
years = np.array([1960, 1980, 2000, 2020, 2040, 2060])
solar = np.array([2, 5, 10, 30, 50, 60]) + np.random.normal(0, 2, 6)
wind = np.array([1, 3, 15, 40, 45, 50]) + np.random.normal(0, 2, 6)
hydro = np.array([10, 20, 25, 27, 30, 33]) + np.random.normal(0, 2, 6)
biomass = np.array([5, 8, 12, 15, 18, 20]) + np.random.normal(0, 2, 6)
benchmark = np.linspace(25, 25, len(years))  # Benchmark line

# Extracted variables
fill_label_solar = "Solar Energy"
fill_label_wind = "Wind Energy"
fill_label_hydro = "Hydro Energy"
fill_label_biomass = "Biomass Energy"
plot_label_benchmark = "Benchmark"
title_text = "Renewable Energy Adoption Over Time"
xlabel_text = "Year"
ylabel_text = "Adoption Rate (%)"
legend_title = "Energy Sources"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
xlim_values = (min(years), max(years))
ylim_values = (0, 70)
xticks_values = years
yticks_values = np.arange(0, 71, 10)
legend_loc = "upper left"
legend_bbox_to_anchor = (1.05, 1)
legend_ncol = 1

# Create the plot with a business-appropriate color scheme
plt.figure(figsize=(10, 6))
plt.fill_between(years, solar, color="#8c564b", alpha=0.6, label=fill_label_solar)  # Earth brown
plt.fill_between(years, wind, color="#2ca02c", alpha=0.6, label=fill_label_wind)  # Calming green
plt.fill_between(years, hydro, color="#1f77b4", alpha=0.6, label=fill_label_hydro)  # Soft blue
plt.fill_between(years, biomass, color="#d62728", alpha=0.6, label=fill_label_biomass)  # Calm red
plt.plot(
    years,
    benchmark,
    color="gray",
    linestyle="--",
    linewidth=2,
    label=plot_label_benchmark,
)

# Add a title and labels with enhanced formatting
plt.title(title_text, fontsize=14, y=1.1)
plt.xlabel(xlabel_text, fontsize=12)
plt.ylabel(ylabel_text, fontsize=12)
plt.xticks(xticks_values)
plt.yticks(yticks_values)
plt.gca().tick_params(axis="both", which="both", length=0)

# Setting the limits explicitly to prevent cut-offs
plt.xlim(*xlim_values)
plt.ylim(*ylim_values)

# Adding a legend with a title
plt.legend(
    title=legend_title,
    frameon=False,
    framealpha=0.8,
    loc=legend_loc,
    bbox_to_anchor=legend_bbox_to_anchor,
    ncol=legend_ncol,
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to ensure no clipping
plt.tight_layout()
plt.savefig("area_55.pdf", bbox_inches="tight")
