
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.ticker as ticker

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# New Data Generation
np.random.seed(42)  # For reproducibility in this example

# Dosage levels
dosage_values = np.linspace(10, 100, 11)
blood_pressure_values = np.random.uniform(120, 180, 11)
heart_rate_values = np.random.uniform(60, 100, 11)
cholesterol_values = np.random.uniform(150, 250, 11)

# Simulated effect rates for 'Medication A' and 'Medication B'
effect_blood_pressure_a = np.random.uniform(-15, 0, 11)
effect_heart_rate_a = np.random.uniform(-10, 5, 11)
effect_cholesterol_a = np.random.uniform(-20, 10, 11)

effect_blood_pressure_b = np.random.uniform(-12, 3, 11)
effect_heart_rate_b = np.random.uniform(-8, 7, 11)
effect_cholesterol_b = np.random.uniform(-15, 15, 11)

# Labels and text parameters
label_medication_a = "Medication A"
label_medication_b = "Medication B"
xlabel_dosage = "Dosage Levels [mg]"
ylabel_effect = "Effect on Health Metric"
xlabel_blood_pressure = "Blood Pressure Level [mmHg]"
xlabel_heart_rate = "Heart Rate [bpm]"
xlabel_cholesterol = "Cholesterol Level [mg/dL]"
suptitle_text = "Comparison of Health Metric Effects by Medications A and B"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
ylim_blood_pressure = (-20, 5)
xlim_blood_pressure = (5, 105)
yticks_blood_pressure = [-20, -15, -10, -5, 0, 5]
xticks_blood_pressure = np.linspace(10, 100, 10)

ylim_heart_rate = (-12, 8)
xlim_heart_rate = (5, 105)
yticks_heart_rate = [-12, -8, -4, 0, 4, 8]
xticks_heart_rate = np.linspace(10, 100, 10)

ylim_cholesterol = (-25, 20)
xlim_cholesterol = (5, 105)
yticks_cholesterol = [-25, -20, -15, -10, -5, 0, 5, 10, 15, 20]
xticks_cholesterol = np.linspace(10, 100, 10)

# Create figure and subplots
fig, axs = plt.subplots(3, 1, figsize=(6, 10))
fig.suptitle(suptitle_text, fontsize=16)

# Color scheme for the bars
color_medication_a = "#1f77b4"
color_medication_b = "#ff7f0e"

# Top chart - Blood Pressure
axs[0].bar(
    dosage_values - 2,
    effect_blood_pressure_a,
    width=4,
    zorder=10,
    color=color_medication_a,
    label=label_medication_a,
)
axs[0].bar(
    dosage_values + 2,
    effect_blood_pressure_b,
    width=4,
    zorder=10,
    color=color_medication_b,
    label=label_medication_b,
)
axs[0].set_xlabel(xlabel_dosage)
axs[0].set_ylabel(ylabel_effect)
axs[0].set_ylim(ylim_blood_pressure)
axs[0].set_xlim(xlim_blood_pressure)
axs[0].set_yticks(yticks_blood_pressure)
axs[0].xaxis.set_major_locator(ticker.FixedLocator(xticks_blood_pressure))
axs[0].xaxis.set_major_formatter(ticker.FuncFormatter(lambda x, _: f"{x:.0f}"))
offsetticks2 = [5] + [i + 2.5 for i in xticks_blood_pressure]
axs[0].xaxis.set_minor_locator(ticker.FixedLocator(offsetticks2))
axs[0].grid(True, which="minor", axis="x", color="gray", linestyle='--', lw=0.5)
axs[0].grid(True, which="major", axis="y", color="gray", linestyle='-', lw=0.7)
axs[0].tick_params(axis="x", which="major", length=0)

# Middle chart - Heart Rate
axs[1].bar(
    dosage_values - 2,
    effect_heart_rate_a,
    width=4,
    zorder=10,
    color=color_medication_a,
)
axs[1].bar(
    dosage_values + 2,
    effect_heart_rate_b,
    width=4,
    zorder=10,
    color=color_medication_b,
)
axs[1].set_xlabel(xlabel_dosage)
axs[1].set_ylabel(ylabel_effect)
axs[1].set_ylim(ylim_heart_rate)
axs[1].set_xlim(xlim_heart_rate)
axs[1].set_yticks(yticks_heart_rate)
axs[1].xaxis.set_major_locator(ticker.FixedLocator(xticks_heart_rate))
axs[1].xaxis.set_major_formatter(ticker.FuncFormatter(lambda x, _: f"{x:.0f}"))
offsetticks1 = [5] + [i + 2.5 for i in xticks_heart_rate]
axs[1].xaxis.set_minor_locator(ticker.FixedLocator(offsetticks1))
axs[1].grid(True, which="minor", axis="x", color="gray", linestyle='--', lw=0.5)
axs[1].grid(True, which="major", axis="y", color="gray", linestyle='-', lw=0.7)
axs[1].tick_params(axis="x", which="major", length=0)

# Bottom chart - Cholesterol Level
axs[2].bar(
    dosage_values - 2,
    effect_cholesterol_a,
    width=4,
    zorder=10,
    color=color_medication_a,
)
axs[2].bar(
    dosage_values + 2,
    effect_cholesterol_b,
    width=4,
    zorder=10,
    color=color_medication_b,
)
axs[2].set_xlabel(xlabel_dosage)
axs[2].set_ylabel(ylabel_effect)
axs[2].set_ylim(ylim_cholesterol)
axs[2].set_xlim(xlim_cholesterol)
axs[2].set_yticks(yticks_cholesterol)
axs[2].tick_params(axis="x", which="major", length=0)

axs[2].xaxis.set_major_locator(ticker.FixedLocator(xticks_cholesterol))
axs[2].xaxis.set_major_formatter(ticker.FuncFormatter(lambda x, _: f"{x:.0f}"))
offsetticks = [5] + [i + 2.5 for i in xticks_cholesterol]
axs[2].xaxis.set_minor_locator(ticker.FixedLocator(offsetticks))
axs[2].grid(True, which="minor", axis="x", color="gray", linestyle='--', lw=0.5)
axs[2].grid(True, which="major", axis="y", color="gray", linestyle='-', lw=0.7)

# Add legend
fig.legend(loc="upper center", ncol=2, bbox_to_anchor=(0.5, 1.03), frameon=False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout(rect=[0, 0, 1, 0.97])
plt.savefig("bar_144.pdf", bbox_inches="tight")
