
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Data
genres = ["Action", "Comedy", "Drama", "Sci-Fi", "Romance"]
ticket_sales = {
    "Q1": [150, 200, 250, 180, 220],
    "Q2": [180, 220, 300, 240, 260],
    "Q3": [210, 240, 350, 280, 300],
    "Q4": [190, 230, 320, 260, 280],
}

# Plot labels and types
ylabel_text = "Ticket Sales (in thousands)"
xlabel_text = "Film Genres"
title_text = "Quarterly Ticket Sales for Different Film Genres"
legend_title = "Quarter"

# Plot limits and ticks
ylim_values = (0, 400)
yticks_values = [0, 100, 200, 300, 400]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
bar_label_fontsize = 10
ylabel_fontsize = 12

# Colors for the bars
colors = ["#8FBC8F", "#DAA520", "#556B2F", "#8B4513", "#DEB887"]

# Setup the figure and axes
fig, ax = plt.subplots(figsize=(12, 4))

# Bar width
bar_width = 0.1

# Positions of the bars on the x-axis
r = np.arange(len(genres))

# Create bars for each quarter
for i, (quarter, values) in enumerate(ticket_sales.items()):
    ax.bar(
        r + i * bar_width,
        values,
        color=colors[i],
        width=bar_width,
        edgecolor="white",
        label=quarter,
    )

# Add labels on top of the bars
for i, (quarter, values) in enumerate(ticket_sales.items()):
    for j, value in enumerate(values):
        ax.text(
            j + i * bar_width,
            value + 5,
            str(value),
            ha="center",
            va="bottom",
            fontsize=bar_label_fontsize,
        )

# General layout
ax.set_ylabel(ylabel_text, fontsize=ylabel_fontsize)
ax.set_xticks(r + bar_width * (len(ticket_sales) - 1) / 2)
ax.set_xticklabels(genres)
ax.set_ylim(*ylim_values)
ax.set_yticks(yticks_values)
ax.legend(
    title=legend_title,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.2),
    frameon=False,
    ncol=5,
)
ax.set_facecolor("#eaeaf2")
ax.yaxis.grid(True, color="white")
ax.set_axisbelow(True)

# Remove spines
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["bottom"].set_visible(False)
ax.spines["left"].set_visible(False)

plt.tick_params(axis="both", which="both", length=0)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(12, 4)
plt.tick_params(axis="both", which="both", length=0)
plt.tight_layout()
plt.savefig("bar_191.pdf", bbox_inches="tight")
