
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data
genres = [
    "Action",
    "Comedy",
    "Drama",
    "Fantasy",
    "Horror",
    "Mystery",
    "Romance",
    "Thriller",
    "Western",
    "Sci-Fi",
    "Documentary",
    "Animation",
    "Musical",
    "Family",
    "Adventure",
    "War",
    "Crime",
    "Sport",
    "Biography",
    "History",
    "Film-Noir",
    "Game-Show",
    "Reality-TV",
    "Talk-Show",
    "News"
]
delta_viewership = np.random.uniform(-15, 10, len(genres))

# Styling variables
xlim_values = (-1, len(genres))
ylim_values = (-15, 11)
xlabel_text = "Media Genres"
ylabel_text = "ΔViewership % (With Campaign) - Viewership % (Without Campaign)"
title_text = "Impact of Marketing Campaign on Media Genres"
legend_labels = ["ΔViewership < -5%", "-5% <= ΔViewership <= 5%", "ΔViewership > 5%"]
yticks_values = [-15, -10, -5, 0, 5, 10]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plot
fig, ax = plt.subplots(figsize=(10, 6))

# Colors based on delta viewership
colors = [
    "#1f77b4" if x < -5 else "#ff7f0e" if -5 <= x <= 5 else "#2ca02c"
    for x in delta_viewership
]
bars = ax.bar(genres, delta_viewership, color=colors)

# Labels and Title
ax.set_xlim(*xlim_values)
ax.set_xticks([])
ax.set_ylim(*ylim_values)
ax.set_yticks(yticks_values)
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.set_title(title_text)

# Add text labels
for bar, genre in zip(bars, genres):
    y = bar.get_height()
    rotation = 90
    if y < 0:
        ax.text(
            bar.get_x() + bar.get_width() / 2,
            0.2,
            genre,
            rotation=rotation,
            ha="center",
            va="bottom",
            fontsize=8
        )
    else:
        ax.text(
            bar.get_x() + bar.get_width() / 2,
            -0.2,
            genre,
            rotation=rotation,
            ha="center",
            va="top",
            fontsize=8
        )

# Legend
blue_patch = plt.Rectangle((0, 0), 1, 1, fc="#1f77b4", edgecolor="none")
orange_patch = plt.Rectangle((0, 0), 1, 1, fc="#ff7f0e", edgecolor="none")
green_patch = plt.Rectangle((0, 0), 1, 1, fc="#2ca02c", edgecolor="none")
ax.legend(
    [blue_patch, orange_patch, green_patch],
    legend_labels,
    loc="lower center",
    bbox_to_anchor=(0.5, -0.15),
    ncol=3,
    frameon=False,
)

plt.tick_params(axis="both", which="both", length=0)
ax.grid(axis='y', linestyle='--', alpha=0.7)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout
plt.tight_layout()
plt.savefig("bar_206.pdf", bbox_inches="tight")
