
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.colors as mcolors

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Data
categories = [
    "Residential",
    "Commercial",
    "Industrial",
    "Transportation",
    "Agriculture",
    "Public Services",
    "Others",
]

differences = np.random.randint(-500, 501, len(categories))
title = "Relative Difference in Energy Consumption Across Sectors"
xlabel = "Difference (GWh)"
ylabel = "Energy Sector"
xlim = [-500, 500]
xticks = range(-500, 501, 100)
cbarlabels = ["Decrease in Consumption", "Increase in Consumption"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
colors = [
    (
        plt.cm.Blues(np.array(i) / min(differences))
        if i < 0
        else plt.cm.Purples(np.array(i) / max(differences))
    )
    for i in differences
]

# Create figure and plot
fig, ax = plt.subplots(figsize=(12, 6))
bars = ax.barh(categories, differences, color=colors)

# Set title and labels
ax.set_title(title, fontsize=16, fontweight='bold')
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)

# Set x-axis limits and labels
ax.set_xlim(xlim)
ax.set_xticks(xticks)
ax.xaxis.grid(True)

# Annotate the bars with their respective differences
for bar in bars:
    width = bar.get_width()
    label_x_pos = width - (np.sign(width) * 10)
    ax.text(
        label_x_pos, 
        bar.get_y() + bar.get_height()/2, 
        f'{int(width)} GWh', 
        va='center', 
        ha='center',
        color='white' if abs(width) > 50 else 'black',
        fontweight='bold'
    )

# Create colorbars
cbar1 = plt.colorbar(
    plt.cm.ScalarMappable(norm=mcolors.Normalize(0, 500), cmap="Blues"),
    ax=ax,
    orientation="vertical",
    pad=0.01,
    aspect=20,
)
cbar2 = plt.colorbar(
    plt.cm.ScalarMappable(norm=mcolors.Normalize(0, 500), cmap="Purples"),
    ax=ax,
    orientation="vertical",
    pad=0.01,
    aspect=20,
)
cbar1.set_label(cbarlabels[0], fontsize=12)
cbar2.set_label(cbarlabels[1], fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Show plot with tight layout
plt.tight_layout()

# Save the figure
plt.savefig("bar_257.pdf", bbox_inches="tight")
