

# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt
import colorsys

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Redefining the data
cities = ["Tokyo", "Delhi", "Shanghai", "São Paulo", "Mumbai", "Beijing", "Cairo"]
populations = [37.4, 30.9, 27.1, 22.0, 20.4, 20.0, 19.2]

# Sorting the data in descending order while keeping track of the models order
sorted_data = sorted(zip(populations, cities), reverse=True)
sorted_populations, sorted_cities = zip(*sorted_data)

title = "World Cities Population"
xlabel = "City"
ylabel = "Population (Millions)"
ylim = [0, np.max(sorted_populations) + 10]


# Generate colors inspired by art styles (e.g., pastel colors)
def hex_to_rgb(hex):
    hex = hex.lstrip('#')
    return tuple(int(hex[i:i+2], 16) / 255.0 for i in (0, 2, 4))



# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and bar chart with the sorted data
plt.figure(figsize=(12, 8))
color_palette = ["#FFB7C5", "#FFE5B4", "#D3FFCE", "#B0E0E6", "#C9C0BB", "#FFCCCB", "#F5FFFA"]
colors = [hex_to_rgb(color) for color in color_palette]

bars = plt.bar(sorted_cities, sorted_populations, color=colors)

# Randomly decide where to put the text based on the value of the bar
for bar in bars:
    yval = bar.get_height()
    text_y = (
        yval - 5 if yval > 10 else yval + 1
    )  # Slight modification to avoid negative values
    plt.text(
        bar.get_x() + bar.get_width() / 2,
        text_y,
        f"{yval}M",
        ha="center",
        va="top" if text_y < yval else "bottom",
        fontsize=12, fontweight='bold', color='black'
    )

# Set chart title and labels
plt.title(title, fontsize=16, fontweight='bold')
plt.xlabel(xlabel, fontsize=14)
plt.ylabel(ylabel, fontsize=14)

# Set y-axis range to a bit higher than the max value
plt.ylim(ylim)

# Set gridlines and ticks
plt.grid(axis="y", linestyle="--", alpha=0.7)

# Set tick rotation
plt.xticks(rotation=45, ha='right', fontsize=12)

# Hide the top and right spines
plt.gca().spines["top"].set_visible(False)
plt.gca().spines["right"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Apply tight layout
plt.tight_layout()

plt.savefig("bar_281.pdf", bbox_inches="tight")

