
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)  # Seed for reproducibility

# Expanded categories with descriptive names
categories = [
    "Physics",
    "Chemistry",
    "Biology",
    "Mathematics",
    "Computer Science",
    "Engineering",
    "Medicine",
]

# Increased number of layers with descriptive names
layer_data = {
    f"Institute {chr(65+i)} Papers": np.random.randint(50, 200, size=len(categories))
    for i in range(6)
}

title = "Distribution of Research Papers by Field and Institute"
xlabel = "Scientific Fields"
ylabel = "Number of Papers"

# Hatch patterns to add texture to the bars
hatches = ["//", "\\\\", "||", "--", "++", "xx"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Color palette with diverse colors
colors = ["#a6cee3", "#1f78b4", "#b2df8a", "#33a02c", "#fb9a99", "#e31a1c"]

fig, ax = plt.subplots(figsize=(10, 7))

bottoms = np.array([0] * len(categories))

for i, (layer, values) in enumerate(layer_data.items()):
    bars = ax.bar(
        categories,
        values,
        bottom=bottoms,
        color=colors[i],
        label=layer,
        hatch=hatches[i],
    )

    # Text styling for better readability
    for bar, bottom in zip(bars, bottoms):
        height = bar.get_height()
        ax.annotate(
            f'{height}',
            xy=(bar.get_x() + bar.get_width() / 2, bottom + height / 2),
            ha='center',
            va='center',
            fontsize=10,
            color='black',
            weight='bold'
        )
    # Update the bottoms for stacking
    bottoms += values

# Chart title and labels with enhanced styles
plt.title(title, fontsize=16, fontweight='bold')
plt.xlabel(xlabel, fontsize=12, fontweight='bold')
plt.ylabel(ylabel, fontsize=12, fontweight='bold')

# Adjust legend to be at the top outside the plot area
plt.legend(
    title="Institutes",
    title_fontsize="13",
    fontsize="11",
    loc="upper center",
    bbox_to_anchor=(0.5, 1.25),
    ncol=3,
)

# Adding grid lines for better readability
ax.yaxis.grid(True, linestyle="--", which="major", color="grey", alpha=0.6)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("bar_287.pdf", bbox_inches="tight")
