

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Sample data for economic study
data1 = [
    np.random.normal(5, std, 50)
    for std in [1, 2, 1.5, 1, 2]
]
data2 = [
    np.random.normal(7, std, 50)
    for std in [1.5, 2.5, 2, 1.5, 2.5]
]
data3 = [
    np.random.normal(10, std, 50)
    for std in [2, 3, 2.5, 2, 3.5]
]

# Text label parameters
labels = [
    "Investment Return",
    "Low Yield",
    "Conservative Portfolio",
    "Moderate Portfolio",
    "Aggressive Portfolio",
    "Benchmark"
]
xlabel = "Investment Period"
ylabel = "Annual Return (%)"
ylim = [0, 20]

axhlines = [5, 15]  # Return levels considered low and high
title = "Performance of Different Investment Portfolios"
supertitle = "Economic Study of Investment Returns"
xticklabels=["Year 1", "Year 3", "Year 5", "Year 7", "Year 10"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

bp1 = ax.boxplot(
    data1,
    positions=np.array(range(len(data1))) * 2.0 - 0.4,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)
bp2 = ax.boxplot(
    data2,
    positions=np.array(range(len(data2))) * 2.0,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)
bp3 = ax.boxplot(
    data3,
    positions=np.array(range(len(data3))) * 2.0 + 0.4,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)

# Set properties for each boxplot
for bp, color in zip([bp1, bp2, bp3], ["#6a5acd", "#32cd32", "#ff4500"]):
    for patch in bp["boxes"]:
        patch.set_facecolor(color)
    for whisker in bp["whiskers"]:
        whisker.set(color="gray", linewidth=1)
    for cap in bp["caps"]:
        cap.set(color="gray", linewidth=1)
    for median in bp["medians"]:
        median.set(color="black", linewidth=2)

# Get the bottom and height of the boxes in bp2
box_data = [
    np.abs(box.get_path().vertices[1][1] - box.get_path().vertices[2][1])
    for box in bp2["boxes"]
]

max_box = np.max(box_data) 
min_box = np.min(box_data) 
max_pos = np.argmax(box_data)
min_pos = np.argmin(box_data)

ax.plot(
    max_pos * 2.0,
    bp2["medians"][max_pos].get_ydata()[0],
    marker="x",
    color="darkred",
    markersize=10,
)
ax.plot(
    min_pos * 2.0,
    bp2["medians"][min_pos].get_ydata()[0],
    marker="x",
    color="darkred",
    markersize=10,
)

# Add dashed lines for return level thresholds
ax.axhline(y=axhlines[0], color="red", linestyle="--", label=labels[1])
ax.axhline(y=axhlines[1], color="red", linestyle="--", label=labels[1])

# Create Legend
axhline_legend = mlines.Line2D([], [], color="red", linestyle="--", label=labels[1])
marker_legend = mlines.Line2D([], [], color="darkred", marker="x", linestyle="None", label="Outliers")

patch1 = mpatches.Patch(color="#6a5acd", label=labels[2])
patch2 = mpatches.Patch(color="#32cd32", label=labels[3])
patch3 = mpatches.Patch(color="#ff4500", label=labels[4])

ax.legend(
    handles=[patch1, patch2, patch3, axhline_legend, marker_legend],
    loc="upper center",
    ncol=3,
    frameon=False,
)

ax.set_title(title, pad=20)
fig.suptitle(supertitle, fontsize=14, fontweight='bold')

ax.set_ylim(ylim)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

ax.set_xticks(range(0, len(data1) * 2, 2))
ax.set_xticklabels(xticklabels)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout(rect=[0, 0.03, 1, 0.95])
plt.savefig("box_38.pdf", bbox_inches="tight")

