# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Generate new sample data for population growth over years
data1 = [
    np.random.normal(500, std, 50)
    for (i, std) in enumerate(np.random.choice(range(50, 150), 7, replace=False))
]
data2 = [
    np.random.normal(600, std, 50)
    for (i, std) in enumerate(np.random.choice(range(50, 150), 7, replace=False))
]
data3 = [
    np.random.normal(700, std, 50)
    for (i, std) in enumerate(np.random.choice(range(50, 150), 7, replace=False))
]
labels = [
    "Population Threshold",
    "City with Max Growth",
    "City A",
    "City B",
    "City C",
]
xlabel = "Years"
ylabel = "Population (in thousands)"
ylim = [200, 1000]
axhlines = [300, 900]
title = "Population Growth Over Years for Various Cities"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Boxplot
fig, ax = plt.subplots(
    figsize=(10, 8)
)  # Adjust figure size to match original dimensions
bp1 = ax.boxplot(
    data1,
    positions=np.array(range(len(data1))) * 2.0 - 0.4,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)
bp2 = ax.boxplot(
    data2,
    positions=np.array(range(len(data2))) * 2.0,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)
bp3 = ax.boxplot(
    data3,
    positions=np.array(range(len(data3))) * 2.0 + 0.4,
    widths=0.3,
    patch_artist=True,
    showfliers=False,
)

# Set properties for each boxplot
for bp, color in zip([bp1, bp2, bp3], ["#1f77b4", "#ff7f0e", "#2ca02c"]):
    for patch in bp["boxes"]:
        patch.set_facecolor(color)
    for whisker in bp["whiskers"]:
        whisker.set(color="gray", linewidth=1.5)
    for cap in bp["caps"]:
        cap.set(color="gray", linewidth=1.5)
    for median in bp["medians"]:
        median.set(color="black", linewidth=2)
    # for flier in bp['fliers']:
    #     flier.set(marker='x', color='black', alpha=0.5)

# Get the bottom and height of the boxes in bp2
box_data = [
    np.abs(box.get_path().vertices[1][1] - box.get_path().vertices[2][1])
    for box in bp2["boxes"]
]

# Find the maximum and minimum values of the boxes
max_box = np.max(box_data)  # The top of the box is h
min_box = np.min(box_data)  # The bottom of the box is y

# Find the positions of the maximum and minimum boxes
max_pos = np.argmax(box_data)
min_pos = np.argmin(box_data)

# Add cross markers at the maximum and minimum values
ax.plot(
    max_pos * 2.0,
    bp2["medians"][max_pos].get_ydata()[0],
    marker="x",
    color="black",
    markersize=10,
)
ax.plot(
    min_pos * 2.0,
    bp2["medians"][min_pos].get_ydata()[0],
    marker="x",
    color="red",
    markersize=10,
)

# Add dashed line for target thresholds
ax.axhline(y=axhlines[0], color="blue", linestyle="--", label="Lower Threshold")
ax.axhline(y=axhlines[1], color="green", linestyle="--", label="Upper Threshold")

# Add legend
# Create a Line2D instance for the axhline legend
axhline_legend1 = mlines.Line2D([], [], color="blue", linestyle="--", label=labels[0])
axhline_legend2 = mlines.Line2D([], [], color="green", linestyle="--", label=labels[0])
marker_legend_max = mlines.Line2D(
    [], [], color="black", marker="x", linestyle="None", label=labels[1]
)
marker_legend_min = mlines.Line2D(
    [], [], color="red", marker="x", linestyle="None", label="City with Min Growth"
)
# Add the legends to the plot
patch1 = mpatches.Patch(color="#1f77b4", label=labels[2])
patch2 = mpatches.Patch(color="#ff7f0e", label=labels[3])
patch3 = mpatches.Patch(color="#2ca02c", label=labels[4])

ax.legend(
    handles=[
        patch1,
        patch2,
        patch3,
        axhline_legend1,
        axhline_legend2,
        marker_legend_max,
        marker_legend_min,
    ],
    loc="upper right",
    ncol=1,
    frameon=True,
)
# Set x and y axis labels
ax.set_ylim(ylim)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Set x-axis tick positions and labels
ax.set_xticks(range(0, len(data1) * 2, 2))
ax.set_xticklabels(
    ["Year 1", "Year 2", "Year 3", "Year 4", "Year 5", "Year 6", "Year 7"]
)

# Add chart title
plt.title(title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("box_40.pdf", bbox_inches="tight")
