

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Simulated interpretation scores for 10 different philosophical texts
data = [np.random.normal(loc, 10, 100) for loc in range(60, 110, 5)]
xticklabels = [
    "Plato's Republic", "Aristotle's Nicomachean Ethics", "Kant's Critique of Pure Reason",
    "Nietzsche's Beyond Good and Evil", "Sartre's Being and Nothingness",
    "Heidegger's Being and Time", "Descartes' Meditations", "Hume's Enquiry", 
    "Spinoza's Ethics", "Locke's Second Treatise"
]
xlabel = "Philosophical Texts"
ylabel = "Interpretation Scores"
title = "Distribution of Interpretation Scores Across Different Texts"
suptitle = "Philosophical Texts Interpretation Analysis"
min_annotation = "Lowest Score"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the boxplot
fig, ax = plt.subplots(figsize=(10, 6))

# Customizing boxplot appearance
boxprops = dict(linestyle="-.", linewidth=1.5, color="green", facecolor="lightgreen")
medianprops = dict(linestyle="-", linewidth=1.5, color="purple")
meanprops = dict(marker="D", markerfacecolor="yellow", markersize=7)

bp = ax.boxplot(
    data,
    patch_artist=True,
    showmeans=True,
    meanprops=meanprops,
    showfliers=False,
    boxprops=boxprops,
    medianprops=medianprops,
)

# Modify additional box properties
for patch in bp['boxes']:
    patch.set(edgecolor="darkgreen")

# Adding x-axis labels with a slant for better readability
ax.set_xticklabels(xticklabels, rotation=45, fontsize=10)

# Setting axis labels and title
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_title(title, fontsize=14, fontweight='bold')
fig.suptitle(suptitle, fontsize=16, fontweight='bold')

# Add markers and annotations for minimum values
for i, line in enumerate(bp["whiskers"][::2]):
    mid_val = line.get_ydata()[1]
    ax.plot(i + 1, mid_val, marker="o", color="red")
    ax.text(i + 1, mid_val, f'{min_annotation}', ha='center', fontsize=8, color='red')

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout(rect=[0, 0, 1, 0.96])
plt.savefig("box_74.pdf", bbox_inches="tight")

