
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.patches import Patch

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Simulated data generation for signal strength and network traffic
signal_strength = np.linspace(-100, 0, 400)
network_traffic = np.linspace(0, 100, 400)
SIGNAL_STRENGTH, NETWORK_TRAFFIC = np.meshgrid(signal_strength, network_traffic)
pos = np.dstack((SIGNAL_STRENGTH, NETWORK_TRAFFIC))

# Simulate Gaussian distributions for signal strength and network traffic
def gaussian(x, y, mean, cov):
    return np.exp(
        -((x - mean[0]) ** 2 / (2 * cov[0][0]) + (y - mean[1]) ** 2 / (2 * cov[1][1]))
    )

Z_signal_strength = gaussian(SIGNAL_STRENGTH, NETWORK_TRAFFIC, [-50, 50], [[200, 0], [0, 500]])
Z_network_traffic = gaussian(SIGNAL_STRENGTH, NETWORK_TRAFFIC, [-30, 70], [[150, 0], [0, 300]])

# Extracted variables
title = "Distribution of Signal Strength and Network Traffic"
labels = ["Signal Strength", "Network Traffic"]
xlabel = "Signal Strength (dBm)"
ylabel = "Network Traffic (Mbps)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting
plt.figure(figsize=(10, 6))

# Contour plots for the distributions
contour_signal_strength = plt.contourf(SIGNAL_STRENGTH, NETWORK_TRAFFIC, Z_signal_strength, cmap="Reds", alpha=0.6)
contour_network_traffic = plt.contourf(SIGNAL_STRENGTH, NETWORK_TRAFFIC, Z_network_traffic, cmap="Greens", alpha=0.6)
plt.title(title)

# Create legend with color patches
legend_patches = [
    Patch(color="red", label=labels[0]),
    Patch(color="green", label=labels[1]),
]
plt.legend(handles=legend_patches)

# Axis labels
plt.xlabel(xlabel)
plt.ylabel(ylabel)

# Adjust plot to be visually appealing
plt.gca().set_aspect("equal", adjustable="box")

# ===================
# Part 4: Saving Output
# ===================
# Reduce whitespace around the plot
plt.tight_layout()
plt.savefig("contour_40.pdf", bbox_inches="tight")
