# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data configuration
np.random.seed(42)

# Grid
x = np.linspace(-50, 50, 300)
y = np.linspace(-50, 50, 300)
X, Y = np.meshgrid(x, y)

# Heat source locations (e.g., cities)
heat_sources = [(20, -20), (-20, 20), (-30, -30), (30, 30), (0, 45)]


# Influence strength function for temperature distribution
def temperature_influence(x, y, points):
    Z = np.zeros_like(x)
    for px, py in points:
        Z += np.exp(-((x - px) ** 2 + (y - py) ** 2) / 150)
    return Z * 30  # Scaling to represent temperature


# Calculate Z as influence strength
Z = temperature_influence(X, Y, heat_sources)

# Text variables
title_text = "Temperature Distribution in a Geographic Area"
xlabel_text = "Longitude (degrees)"
ylabel_text = "Latitude (degrees)"
colorbar_label_text = "Temperature (°C)"
heat_source_label_format = "Heat Source ({},{})"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
legend_location = "upper right"
contour_label_format = "%.1f"
fig, ax = plt.subplots(figsize=(10, 10))

# Contour
cnt = ax.contour(X, Y, Z, cmap="viridis", linewidths=1)
contour_filled = ax.contourf(X, Y, Z, cmap="viridis", alpha=0.7)
ax.clabel(cnt, cnt.levels, inline=True, fontsize=10, fmt=contour_label_format)

# Heat source locations
for px, py in heat_sources:
    ax.plot(px, py, "g^", markersize=12, label=heat_source_label_format.format(px, py))

# Adding color bar
cbar = fig.colorbar(contour_filled, ax=ax)
cbar.set_label(colorbar_label_text)

# Titles and labels
ax.set_title(title_text)
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.legend(loc=legend_location)

# Grid for better readability
ax.grid(True, which="both", linestyle="--", linewidth=0.5)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("contour_70.pdf", bbox_inches="tight")
