
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)  # Set a seed for reproducibility
from scipy.stats import gaussian_kde

# Generate data for the plot
x = np.linspace(0, 14, 15)  # Days (over two weeks)
# Simulated daily average temperatures for seven different weather stations
y = [
    np.random.uniform(15, 35) + np.sin(0.5 * (x - i)) * 5
    for index, i in enumerate(np.linspace(0, 6, 7))
]

# Combine all station data into a single array for density estimation
all_data = np.concatenate(y)

# Extracted text variables
xlabel_text = "Average Daily Temperature (°C)"
ylabel_text = "Density"
title_text = "Density of Average Daily Temperatures Over Two Weeks"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Perform Kernel Density Estimation (KDE) using scipy
kde = gaussian_kde(all_data, bw_method='scott')
x_dens = np.linspace(min(all_data), max(all_data), 1000)
density = kde(x_dens)

# Plot the density curve
ax.plot(x_dens, density, color='green', lw=2)
ax.fill_between(x_dens, density, color='green', alpha=0.3)  # Fill under the curve

# Set labels and title
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.set_title(title_text)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["bottom"].set_visible(True)
ax.spines["left"].set_visible(True)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)

# Add grid for readability
plt.grid(True, linestyle=":", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Display the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_38.pdf", bbox_inches="tight")
